from fastapi import Depends, HTTPException, APIRouter
from sqlalchemy.orm import Session

from ..database import session_provider
from ..models.report import ReportInDB, Report, ReportCreate

router = APIRouter()


@router.put("/report/{report_id}", tags=['Report'], response_model=ReportInDB)
def update_report(report_id: int, report_update: ReportCreate, db: Session = Depends(session_provider.get_session)):
    report_db: Report = db.query(Report).filter(Report.id == report_id).first()
    if not report_db:
        raise HTTPException(status_code=404, detail="Report not found")
    for key, value in report_update.model_dump().items():
        setattr(report_db, key, value)
    db.merge(report_db)
    db.commit()
    return ReportInDB.model_validate(report_db)


@router.patch("/report/{report_id}", tags=['Report'], response_model=ReportInDB)
def patch_report(report_id: int, report_update: ReportCreate.model_as_partial(), db: Session = Depends(session_provider.get_session)):
    report_db: Report = db.query(Report).filter(Report.id == report_id).first()
    if not report_db:
        raise HTTPException(status_code=404, detail="Report not found")
    for key, value in report_update.model_dump().items():
        if hasattr(report_db, key) and value is not None:
            setattr(report_db, key, value)
    db.merge(report_db)
    db.commit()
    return ReportInDB.model_validate(report_db)


@router.get("/report/{report_id}", tags=['Report'], response_model=ReportInDB)
def get_report(report_id: int, db: Session = Depends(session_provider.get_session)):
    report_db: Report = db.query(Report).filter(Report.id == report_id).first()
    if not report_db:
        raise HTTPException(status_code=404, detail="Report not found")
    return ReportInDB.model_validate(report_db)


# @router.get("/report", tags=['Report'], response_model=list[ReportInDB])
# def search_report(db: Session = Depends(get_db)):
#     report_all = db.query(Report).all()
#     return [ReportInDB.from_orm(m) for m in report_all]


@router.post("/report", tags=['Report'], response_model=ReportInDB)
def create_report(report_create: ReportCreate, db: Session = Depends(session_provider.get_session)):
    db_report = Report(**report_create.model_dump())
    db.add(db_report)
    db.commit()
    db.refresh(db_report)
    return ReportInDB.model_validate(db_report)
