
#include "config.h"
#include <glib/gi18n.h>
#include <glib.h>
#include <gio/gio.h>
#include <discident-glib.h>
#include <discident-glib-private.h>

static gboolean option_async = FALSE;
static gboolean option_file_list = FALSE;
static guint num_queries = 0;
static GMainLoop *loop = NULL;

static void
test_hash (void)
{
	/* From http://github.com/kastner/dvd_fingerprint/blob/master/dvd_fingerprint.rb */
	g_assert_cmpstr (generate_hash (":/VIDEO_TS/VIDEO_TS.BUP:12288:/VIDEO_TS/VIDEO_TS.IFO:12288:/VIDEO_TS/VIDEO_TS.VOB:58052608:/VIDEO_TS/VTS_01_0.BUP:98304:/VIDEO_TS/VTS_01_0.IFO:98304:/VIDEO_TS/VTS_01_0.VOB:75163648:/VIDEO_TS/VTS_01_1.VOB:1073598464:/VIDEO_TS/VTS_01_2.VOB:1073448960:/VIDEO_TS/VTS_01_3.VOB:1073741824:/VIDEO_TS/VTS_01_4.VOB:1073647616:/VIDEO_TS/VTS_01_5.VOB:835813376", -1), ==, "2075AB92-06CD-ED43-A753-2B75627BE844");
}

static void
test_file_list (void)
{
	guint i;
	struct {
		const char *dir;
		const char *gtin;
		const char *file_list;
	} directories[] = {
		{ "Alexander Nevsky", "3809DA3F-8323-2E48-70C6-861B34968674", ":/VIDEO_TS/VIDEO_TS.BUP:18432:/VIDEO_TS/VIDEO_TS.IFO:18432:/VIDEO_TS/VIDEO_TS.VOB:178176:/VIDEO_TS/VTS_01_0.BUP:24576:/VIDEO_TS/VTS_01_0.IFO:24576:/VIDEO_TS/VTS_01_0.VOB:104853504:/VIDEO_TS/VTS_01_1.VOB:13611008:/VIDEO_TS/VTS_02_0.BUP:18432:/VIDEO_TS/VTS_02_0.IFO:18432:/VIDEO_TS/VTS_02_0.VOB:178176:/VIDEO_TS/VTS_02_1.VOB:25085952:/VIDEO_TS/VTS_03_0.BUP:18432:/VIDEO_TS/VTS_03_0.IFO:18432:/VIDEO_TS/VTS_03_0.VOB:178176:/VIDEO_TS/VTS_03_1.VOB:9498624:/VIDEO_TS/VTS_04_0.BUP:18432:/VIDEO_TS/VTS_04_0.IFO:18432:/VIDEO_TS/VTS_04_0.VOB:178176:/VIDEO_TS/VTS_04_1.VOB:2146304:/VIDEO_TS/VTS_05_0.BUP:65536:/VIDEO_TS/VTS_05_0.IFO:65536:/VIDEO_TS/VTS_05_0.VOB:178176:/VIDEO_TS/VTS_05_1.VOB:1073565696:/VIDEO_TS/VTS_05_2.VOB:1073565696:/VIDEO_TS/VTS_05_3.VOB:1073565696:/VIDEO_TS/VTS_05_4.VOB:1073565696:/VIDEO_TS/VTS_05_5.VOB:206231552" },
		{ "Être et Avoir", "D2740096-2507-09FC-EE0F-D577CBF98536", ":/VIDEO_TS/VIDEO_TS.BUP:22528:/VIDEO_TS/VIDEO_TS.IFO:22528:/VIDEO_TS/VIDEO_TS.VOB:157696:/VIDEO_TS/VTS_01_0.BUP:24576:/VIDEO_TS/VTS_01_0.IFO:24576:/VIDEO_TS/VTS_01_0.VOB:58773504:/VIDEO_TS/VTS_01_1.VOB:9805824:/VIDEO_TS/VTS_02_0.BUP:18432:/VIDEO_TS/VTS_02_0.IFO:18432:/VIDEO_TS/VTS_02_0.VOB:157696:/VIDEO_TS/VTS_02_1.VOB:3958784:/VIDEO_TS/VTS_03_0.BUP:18432:/VIDEO_TS/VTS_03_0.IFO:18432:/VIDEO_TS/VTS_03_0.VOB:157696:/VIDEO_TS/VTS_03_1.VOB:80705536:/VIDEO_TS/VTS_04_0.BUP:18432:/VIDEO_TS/VTS_04_0.IFO:18432:/VIDEO_TS/VTS_04_0.VOB:157696:/VIDEO_TS/VTS_04_1.VOB:99594240:/VIDEO_TS/VTS_05_0.BUP:18432:/VIDEO_TS/VTS_05_0.IFO:18432:/VIDEO_TS/VTS_05_0.VOB:157696:/VIDEO_TS/VTS_05_1.VOB:71680000:/VIDEO_TS/VTS_06_0.BUP:18432:/VIDEO_TS/VTS_06_0.IFO:18432:/VIDEO_TS/VTS_06_0.VOB:157696:/VIDEO_TS/VTS_06_1.VOB:148228096:/VIDEO_TS/VTS_07_0.BUP:18432:/VIDEO_TS/VTS_07_0.IFO:18432:/VIDEO_TS/VTS_07_0.VOB:157696:/VIDEO_TS/VTS_07_1.VOB:87965696:/VIDEO_TS/VTS_08_0.BUP:26624:/VIDEO_TS/VTS_08_0.IFO:26624:/VIDEO_TS/VTS_08_0.VOB:157696:/VIDEO_TS/VTS_08_1.VOB:426055680:/VIDEO_TS/VTS_09_0.BUP:73728:/VIDEO_TS/VTS_09_0.IFO:73728:/VIDEO_TS/VTS_09_0.VOB:157696:/VIDEO_TS/VTS_09_1.VOB:1073565696:/VIDEO_TS/VTS_09_2.VOB:1073565696:/VIDEO_TS/VTS_09_3.VOB:932294656" },
		{ "The Kite Runner", "5F03F0D5-6AB6-FDB4-43AE-825693898CFF", ":/VIDEO_TS/VIDEO_TS.BUP:20480:/VIDEO_TS/VIDEO_TS.IFO:20480:/VIDEO_TS/VIDEO_TS.VOB:935936:/VIDEO_TS/VTS_01_0.BUP:14336:/VIDEO_TS/VTS_01_0.IFO:14336:/VIDEO_TS/VTS_01_1.VOB:471040:/VIDEO_TS/VTS_02_0.BUP:24576:/VIDEO_TS/VTS_02_0.IFO:24576:/VIDEO_TS/VTS_02_0.VOB:13301760:/VIDEO_TS/VTS_02_1.VOB:423208960:/VIDEO_TS/VTS_03_0.BUP:18432:/VIDEO_TS/VTS_03_0.IFO:18432:/VIDEO_TS/VTS_03_0.VOB:229376:/VIDEO_TS/VTS_03_1.VOB:75370496:/VIDEO_TS/VTS_04_0.BUP:40960:/VIDEO_TS/VTS_04_0.IFO:40960:/VIDEO_TS/VTS_04_1.VOB:1073709056:/VIDEO_TS/VTS_04_2.VOB:327958528:/VIDEO_TS/VTS_05_0.BUP:90112:/VIDEO_TS/VTS_05_0.IFO:90112:/VIDEO_TS/VTS_05_0.VOB:58671104:/VIDEO_TS/VTS_05_1.VOB:1073709056:/VIDEO_TS/VTS_05_2.VOB:1073709056:/VIDEO_TS/VTS_05_3.VOB:1073709056:/VIDEO_TS/VTS_05_4.VOB:1073709056:/VIDEO_TS/VTS_05_5.VOB:884168704:/VIDEO_TS/VTS_06_0.BUP:14336:/VIDEO_TS/VTS_06_0.IFO:14336:/VIDEO_TS/VTS_06_1.VOB:43491328:/VIDEO_TS/VTS_07_0.BUP:18432:/VIDEO_TS/VTS_07_0.IFO:18432:/VIDEO_TS/VTS_07_0.VOB:18966528:/VIDEO_TS/VTS_07_1.VOB:6029312" },
		{ "The Straight Story", "4C059E8A-37E4-493E-6272-F9A713DB5AA9", ":/VIDEO_TS/VIDEO_TS.BUP:22528:/VIDEO_TS/VIDEO_TS.IFO:22528:/VIDEO_TS/VIDEO_TS.VOB:157696:/VIDEO_TS/VTS_01_0.BUP:34816:/VIDEO_TS/VTS_01_0.IFO:34816:/VIDEO_TS/VTS_01_0.VOB:255559680:/VIDEO_TS/VTS_01_1.VOB:997376:/VIDEO_TS/VTS_02_0.BUP:65536:/VIDEO_TS/VTS_02_0.IFO:65536:/VIDEO_TS/VTS_02_0.VOB:157696:/VIDEO_TS/VTS_02_1.VOB:1073565696:/VIDEO_TS/VTS_02_2.VOB:1073565696:/VIDEO_TS/VTS_02_3.VOB:1073565696:/VIDEO_TS/VTS_02_4.VOB:1073565696:/VIDEO_TS/VTS_02_5.VOB:1073565696:/VIDEO_TS/VTS_02_6.VOB:1073565696:/VIDEO_TS/VTS_02_7.VOB:376975360:/VIDEO_TS/VTS_03_0.BUP:18432:/VIDEO_TS/VTS_03_0.IFO:18432:/VIDEO_TS/VTS_03_0.VOB:157696:/VIDEO_TS/VTS_03_1.VOB:6483968:/VIDEO_TS/VTS_04_0.BUP:18432:/VIDEO_TS/VTS_04_0.IFO:18432:/VIDEO_TS/VTS_04_0.VOB:157696:/VIDEO_TS/VTS_04_1.VOB:5986304:/VIDEO_TS/VTS_05_0.BUP:18432:/VIDEO_TS/VTS_05_0.IFO:18432:/VIDEO_TS/VTS_05_0.VOB:157696:/VIDEO_TS/VTS_05_1.VOB:15337472:/VIDEO_TS/VTS_06_0.BUP:18432:/VIDEO_TS/VTS_06_0.IFO:18432:/VIDEO_TS/VTS_06_0.VOB:157696:/VIDEO_TS/VTS_06_1.VOB:67874816:/VIDEO_TS/VTS_07_0.BUP:18432:/VIDEO_TS/VTS_07_0.IFO:18432:/VIDEO_TS/VTS_07_0.VOB:157696:/VIDEO_TS/VTS_07_1.VOB:123170816:/VIDEO_TS/VTS_08_0.BUP:18432:/VIDEO_TS/VTS_08_0.IFO:18432:/VIDEO_TS/VTS_08_0.VOB:157696:/VIDEO_TS/VTS_08_1.VOB:9367552" },
		{ "The Woodsman", "724DAC2A-6BB7-21E8-2F85-BCD0A56ED995", ":/VIDEO_TS/VIDEO_TS.BUP:24576:/VIDEO_TS/VIDEO_TS.IFO:24576:/VIDEO_TS/VIDEO_TS.VOB:157696:/VIDEO_TS/VTS_01_0.BUP:92160:/VIDEO_TS/VTS_01_0.IFO:92160:/VIDEO_TS/VTS_01_0.VOB:248268800:/VIDEO_TS/VTS_01_1.VOB:1073565696:/VIDEO_TS/VTS_01_2.VOB:1073565696:/VIDEO_TS/VTS_01_3.VOB:1073565696:/VIDEO_TS/VTS_01_4.VOB:1073565696:/VIDEO_TS/VTS_01_5.VOB:773203968:/VIDEO_TS/VTS_02_0.BUP:30720:/VIDEO_TS/VTS_02_0.IFO:30720:/VIDEO_TS/VTS_02_0.VOB:157696:/VIDEO_TS/VTS_02_1.VOB:743315456:/VIDEO_TS/VTS_03_0.BUP:18432:/VIDEO_TS/VTS_03_0.IFO:18432:/VIDEO_TS/VTS_03_0.VOB:157696:/VIDEO_TS/VTS_03_1.VOB:122218496:/VIDEO_TS/VTS_04_0.BUP:26624:/VIDEO_TS/VTS_04_0.IFO:26624:/VIDEO_TS/VTS_04_0.VOB:157696:/VIDEO_TS/VTS_04_1.VOB:625369088:/VIDEO_TS/VTS_05_0.BUP:26624:/VIDEO_TS/VTS_05_0.IFO:26624:/VIDEO_TS/VTS_05_0.VOB:157696:/VIDEO_TS/VTS_05_1.VOB:592257024:/VIDEO_TS/VTS_06_0.BUP:20480:/VIDEO_TS/VTS_06_0.IFO:20480:/VIDEO_TS/VTS_06_0.VOB:157696:/VIDEO_TS/VTS_06_1.VOB:200243200:/VIDEO_TS/VTS_07_0.BUP:18432:/VIDEO_TS/VTS_07_0.IFO:18432:/VIDEO_TS/VTS_07_0.VOB:157696:/VIDEO_TS/VTS_07_1.VOB:75008000:/VIDEO_TS/VTS_08_0.BUP:18432:/VIDEO_TS/VTS_08_0.IFO:18432:/VIDEO_TS/VTS_08_0.VOB:157696:/VIDEO_TS/VTS_08_1.VOB:103041024:/VIDEO_TS/VTS_09_0.BUP:18432:/VIDEO_TS/VTS_09_0.IFO:18432:/VIDEO_TS/VTS_09_0.VOB:157696:/VIDEO_TS/VTS_09_1.VOB:146837504:/VIDEO_TS/VTS_10_0.BUP:18432:/VIDEO_TS/VTS_10_0.IFO:18432:/VIDEO_TS/VTS_10_0.VOB:157696:/VIDEO_TS/VTS_10_1.VOB:87701504:/VIDEO_TS/VTS_11_0.BUP:18432:/VIDEO_TS/VTS_11_0.IFO:18432:/VIDEO_TS/VTS_11_0.VOB:157696:/VIDEO_TS/VTS_11_1.VOB:4814848:/VIDEO_TS/VTS_12_0.BUP:18432:/VIDEO_TS/VTS_12_0.IFO:18432:/VIDEO_TS/VTS_12_0.VOB:157696:/VIDEO_TS/VTS_12_1.VOB:5820416" },
	};

	if (g_strcmp0 (g_get_user_name (), "hadess") != 0)
		return;

	for (i = 0; i < G_N_ELEMENTS (directories); i++) {
		GFile *dir;
		char *filename;

		filename = g_build_filename ("/home/data/DVDs/", directories[i].dir, NULL);
		dir = g_file_new_for_path (filename);
		g_free (filename);

		g_assert_cmpstr (discident_get_gtin_file (dir, NULL), ==, directories[i].gtin);

		g_object_set_data (G_OBJECT (dir), "wants-file-list", GINT_TO_POINTER (TRUE));

		g_assert_cmpstr (discident_get_gtin_file (dir, NULL), ==, directories[i].file_list);

		g_object_unref (dir);
	}
}

static void
test_json (void)
{
	guint i;
	struct {
		const char *gtin;
		const char *title;
	} dvds[] = {
		{ "3809DA3F-8323-2E48-70C6-861B34968674", NULL },
		{ "5F03F0D5-6AB6-FDB4-43AE-825693898CFF", "The Kite Runner" },
		{ "4C059E8A-37E4-493E-6272-F9A713DB5AA9", "55853 STRAIGHT_STORY" },
		{ "724DAC2A-6BB7-21E8-2F85-BCD0A56ED995", "WOODSMAN" },
		{ "D2740096-2507-09FC-EE0F-D577CBF98536", "ETRE_AVOIR" },
	};

	for (i = 0; i < G_N_ELEMENTS (dvds); i++) {
		g_assert_cmpstr (discident_get_title_for_gtin (dvds[i].gtin, NULL), ==, dvds[i].title);
	}
}

static void
discident_title_print (GObject *source_object,
		      GAsyncResult *res,
		      gpointer user_data)
{
	GFile *directory = G_FILE (source_object);
	GError *error = NULL;
	char *title;

	title = discident_get_title_file_finish (directory, res, &error);
	if (title == NULL) {
		g_message ("Could not get disc ID: %s", error->message);
		g_object_unref (directory);
		g_error_free (error);
		return;
	}
	g_print ("Title: %s\n", title);
	g_free (title);

	num_queries--;
	if (num_queries == 0)
		g_main_loop_quit (loop);
}

static void
discident_gtin_print (GObject *source_object,
		      GAsyncResult *res,
		      gpointer user_data)
{
	GFile *directory = G_FILE (source_object);
	GError *error = NULL;
	char *id;

	id = discident_get_gtin_file_finish (directory, res, &error);
	if (id == NULL) {
		g_message ("Could not get disc ID: %s", error->message);
		g_object_unref (directory);
		g_error_free (error);
		return;
	}
	g_print ("%s\n", id);
	g_free (id);

	num_queries--;
	if (num_queries == 0)
		g_main_loop_quit (loop);
}

static char **uris = NULL;

int main (int argc, char **argv)
{
	GError *error = NULL;
	GOptionContext *context;
	const GOptionEntry entries[] = {
		{ "async", 'a', 0, G_OPTION_ARG_NONE, &option_async, "Use the async API", NULL },
		{ "file-list", 'f', 0, G_OPTION_ARG_NONE, &option_file_list, "Show the file list instead of the GTIN", NULL },
		{ G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_STRING_ARRAY, &uris, NULL, "[URI...]" },
		{ NULL }
	};
	guint i;

	setlocale (LC_ALL, "");
	g_type_init ();
	g_test_init (&argc, &argv, NULL);
	g_test_bug_base ("http://bugzilla.gnome.org/show_bug.cgi?id=");

	/* Parse our own command-line options */
	context = g_option_context_new ("- test parser functions");
	g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);

	if (g_option_context_parse (context, &argc, &argv, &error) == FALSE) {
		g_print ("Option parsing failed: %s\n", error->message);
		return 1;
	}

	if (uris == NULL) {
		g_test_add_func ("/discident/hash", test_hash);
		g_test_add_func ("/discident/file_list", test_file_list);
		g_test_add_func ("/discident/json", test_json);

		return g_test_run ();
	}

	for (i = 0; uris[i] != NULL; i++) {
		GFile *file;
		char *gtin;
		GError *error = NULL;

		file= g_file_new_for_path (uris[i]);
		if (file == NULL) {
			g_message ("Could not convert %s", uris[i]);
			continue;
		}
		g_object_set_data (G_OBJECT (file), "wants-file-list", GINT_TO_POINTER (option_file_list));
		if (!option_async) {
			char *uri, *title;
			gtin = discident_get_gtin_file (file, &error);
			if (gtin == NULL) {
				g_object_unref (file);
				g_message ("Could not get disc ID: %s", error->message);
				g_error_free (error);
				continue;
			}
			uri = g_file_get_uri (file);
			g_object_unref (file);
			g_print ("URI: %s\n", uri);
			g_free (uri);

			g_print ("Disc GTIN: %s\n", gtin);
			g_print ("Query URI: "QUERY_GTIN"\n", gtin);

			title = discident_get_title_for_gtin (gtin, NULL);
			g_free (gtin);
			g_print ("Title: %s\n", title);
			g_free (title);
		} else {
			num_queries++;
			discident_get_gtin_file_async (file, NULL, discident_gtin_print, NULL);
			num_queries++;
			discident_get_title_file_async (file, NULL, discident_title_print, NULL);
			g_object_unref (file);
		}
	}

	if (option_async) {
		loop = g_main_loop_new (NULL, FALSE);
		g_main_loop_run (loop);
	}

	return 0;
}

