/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <libgnomecanvas/gnome-canvas.h>
#include "gailcanvasitem.h"

static void       gail_canvas_item_destroyed                (GObject           *item,
			                                     gpointer          user_data);
static void       gail_canvas_item_class_init               (GailCanvasItemClass *klass);
static AtkObject* gail_canvas_item_get_parent               (AtkObject         *obj);
static void       gail_canvas_item_component_interface_init (AtkComponentIface *iface);
static guint      gail_canvas_item_add_focus_handler        (AtkComponent      *component,
					                     AtkFocusHandler   handler);
static void       gail_canvas_item_get_extents              (AtkComponent      *component,
					                     gint              *x,
					                     gint              *y,
					                     gint              *width,
					                     gint              *height,
					                     AtkCoordType      coord_type);
static gboolean   gail_canvas_item_grab_focus               (AtkComponent      *component);
static void       gail_canvas_item_remove_focus_handler     (AtkComponent      *component,
					                     guint             handler_id);

static GQuark quark_accessible_object = 0;
static AtkObjectClass *parent_class = NULL;

GType
gail_canvas_item_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo tinfo =
      {
        sizeof (GailCanvasItemClass),
        (GBaseInitFunc) NULL, /* base init */
        (GBaseFinalizeFunc) NULL, /* base finalize */
        (GClassInitFunc) gail_canvas_item_class_init, /* class init */
        (GClassFinalizeFunc) NULL, /* class finalize */
        NULL, /* class data */
        sizeof (GailCanvasItem), /* instance size */
        0, /* nb preallocs */
        (GInstanceInitFunc) NULL, /* instance init */
        NULL /* value table */
      };

      static const GInterfaceInfo atk_component_info =
      {
        (GInterfaceInitFunc) gail_canvas_item_component_interface_init,
        (GInterfaceFinalizeFunc) NULL,
        NULL
      };

      type = g_type_register_static (ATK_TYPE_OBJECT,
                                     "GailCanvasItem", &tinfo, 0);
      g_type_add_interface_static (type, ATK_TYPE_COMPONENT,
                                   &atk_component_info);
    }

  return type;
}



AtkObject *
gail_canvas_item_get_accessible (GnomeCanvasItem *item)
{
  AtkObject* accessible;
  GObject *obj;

  obj = G_OBJECT (item);

  /* See if we have a cached accessible for this object */

  accessible = g_object_get_qdata (obj,
				   quark_accessible_object);

  if (!accessible)
    {
      AtkObjectFactory *factory;
      AtkRegistry *default_registry;

      default_registry = atk_get_default_registry ();
      factory = atk_registry_get_factory (default_registry, 
                                          G_OBJECT_TYPE (item));
      accessible = atk_object_factory_create_accessible (factory,
                                                         obj);
      g_object_set_qdata (obj, quark_accessible_object, accessible);
    }
  return accessible;
}

void
gail_canvas_item_init (GailCanvasItem  *gail_item,
		       GnomeCanvasItem *item)
{
  g_return_if_fail (GAIL_IS_CANVAS_ITEM (gail_item));
  g_return_if_fail (GNOME_IS_CANVAS_ITEM (item));

  gail_item->canvas_item = item;
  ATK_OBJECT (gail_item)->layer = ATK_LAYER_MDI;

  /* Connect the handler for the destroy signal for the GnomeCanvasItem */

  g_signal_connect (item,
		    "destroy",
		    G_CALLBACK (gail_canvas_item_destroyed),
		    gail_item);
}


static void
gail_canvas_item_destroyed (GObject  *item,
			    gpointer user_data)
{
  GailCanvasItem *gail_item;
  
  g_return_if_fail (GAIL_IS_CANVAS_ITEM (user_data));
  gail_item = GAIL_CANVAS_ITEM (user_data);
  gail_item->canvas_item = NULL;
  atk_object_notify_state_change (ATK_OBJECT (gail_item), ATK_STATE_DEFUNCT,
                                  TRUE); 
  g_object_unref (G_OBJECT (gail_item));
}

static void
gail_canvas_item_class_init (GailCanvasItemClass *klass)
{
  AtkObjectClass *class = ATK_OBJECT_CLASS (klass);

  parent_class = g_type_class_ref (ATK_TYPE_OBJECT);

  quark_accessible_object = g_quark_from_static_string ("canvas-item-accessible-object");
  class->get_parent = gail_canvas_item_get_parent;
}

static AtkObject *
gail_canvas_item_get_parent (AtkObject *obj)
{
  GailCanvasItem *gail_canvas_item;
  GnomeCanvasItem *item;

  g_return_val_if_fail (GAIL_IS_CANVAS_ITEM (obj), NULL);
  gail_canvas_item = GAIL_CANVAS_ITEM (obj);
  item = gail_canvas_item->canvas_item;
  g_return_val_if_fail (item, NULL);
  if (item->parent)
    return gail_canvas_item_get_accessible (item->parent);
  else
    return gtk_widget_get_accessible (GTK_WIDGET (item->canvas));
}



static void
gail_canvas_item_component_interface_init (AtkComponentIface *iface)
{
  g_return_if_fail (iface != NULL);

  /*
   * Use default implementation for contains and get_position
   */
  iface->contains = NULL;
  iface->get_position = NULL;
  iface->add_focus_handler = gail_canvas_item_add_focus_handler;
  iface->get_extents = gail_canvas_item_get_extents;
  iface->get_size = NULL;
  iface->grab_focus = gail_canvas_item_grab_focus;
  iface->remove_focus_handler = gail_canvas_item_remove_focus_handler;
  iface->set_extents = NULL;
  iface->set_position = NULL;
  iface->set_size = NULL;
}

static guint
gail_canvas_item_add_focus_handler (AtkComponent    *component,
                                    AtkFocusHandler handler)
{
  return g_signal_connect_closure (component, 
                                   "focus-event",
                                   g_cclosure_new (
						   G_CALLBACK (handler), NULL,
						   (GClosureNotify) NULL),
                                   FALSE);
}

static void
gail_canvas_item_get_extents (AtkComponent *component,
                              gint         *x,
                              gint         *y,
                              gint         *width,
                              gint         *height,
                              AtkCoordType coord_type)
{
  GdkWindow *window;
  GailCanvasItem *gail_canvas_item;
  GnomeCanvasItem *item;
  gint root_x, root_y;
  gdouble world_x1, world_y1, world_x2, world_y2;
  gdouble win_x1, win_y1, win_x2, win_y2;

  /* Get the GailCanvasItem */
  g_return_if_fail (GAIL_IS_CANVAS_ITEM(component));
  gail_canvas_item = GAIL_CANVAS_ITEM (component);

  /* Get the GnomeCanvasItem */
  item = gail_canvas_item->canvas_item;
  g_return_if_fail (item);

  /* If this item has no parent canvas, something's broken */
  g_return_if_fail (item->canvas);

  /* Get the GdkWindow for the parent canvas */
  window = GTK_WIDGET(item->canvas)->window;
  g_return_if_fail (window);
  
  /* Get the screen coordinates of the canvas's GdkWindow */
  gdk_window_get_root_origin (window,
			      &root_x, &root_y);

  /* get the world-based bounds of the canvas item */
  gnome_canvas_item_get_bounds (item,
				&world_x1, &world_y1, &world_x2, &world_y2);

  /* Convert them to window-based coordinates */
  gnome_canvas_world_to_window (item->canvas,
				world_x1, world_y1, &win_x1, &win_y1);
  gnome_canvas_world_to_window (item->canvas,
				world_x2, world_y2, &win_x2, &win_y2);

  /* This could be badly broken */
  *x = (gint) win_x1;
  *y = (gint) win_y1;
  *width = (gint) (win_x2-win_x1);
  *height = (gint) (win_y2-win_y1);

  /* If screen coordinates are requested, modify x and y appropriately */
  if (coord_type == ATK_XY_SCREEN)
    {
      *x += root_x;
      *y += root_y;
    }
  return;
}

static gboolean
gail_canvas_item_grab_focus (AtkComponent    *component)
{
  GailCanvasItem *gail_canvas_item;
  GnomeCanvasItem *item;

  g_return_val_if_fail (GAIL_IS_CANVAS_ITEM (component), FALSE);
  gail_canvas_item = GAIL_CANVAS_ITEM (component);
  g_return_val_if_fail (gail_canvas_item->canvas_item, FALSE);
  item = gail_canvas_item->canvas_item;
  gnome_canvas_item_grab_focus (item);
  return TRUE;
}

static void
gail_canvas_item_remove_focus_handler (AtkComponent *component,
                                       guint        handler_id)
{
  g_signal_handler_disconnect (ATK_OBJECT (component), handler_id);
}
