/* schema.cc
 *
 * Copyright (C) 2000-2002 GConfmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gconfmm/schema.h>

namespace Gnome
{

namespace Conf
{

Schema::Schema()
{
  m_pSchema = gconf_schema_new();
}

Schema::Schema(const GConfSchema* pSchema)
{
  m_pSchema = 0;

  if(pSchema)
    m_pSchema = gconf_schema_copy(const_cast<GConfSchema*>(pSchema));
}

Schema::Schema(const Schema& Schema)
{
  m_pSchema = 0;

  operator=(Schema);
}

Schema::~Schema()
{
  clear();
}

Schema& Schema::operator=(const Schema& src)
{
  if(&src != this)
  {
    clear();

    if(src.gobj())
      m_pSchema = gconf_schema_copy(const_cast<GConfSchema*>(src.gobj()));
  }

  return *this;
}

void Schema::clear()
{
  if(m_pSchema)
  {
    gconf_schema_free(m_pSchema);
    m_pSchema = 0;
  }
}

const GConfSchema* Schema::gobj() const
{
  return m_pSchema;
}

GConfSchema* Schema::gobj()
{
  return m_pSchema;
}

void Schema::set_type(GConfValueType type)
{
  gconf_schema_set_type(gobj(), type);
}

void Schema::set_list_type(GConfValueType  type)
{
  gconf_schema_set_list_type(gobj(), type);
}

void Schema::set_car_type(GConfValueType  type)
{
  gconf_schema_set_car_type(gobj(), type);
}

void Schema::set_cdr_type(GConfValueType  type)
{
  gconf_schema_set_cdr_type(gobj(), type);
}

void Schema::set_locale(const std::string& locale)
{
  gconf_schema_set_locale(gobj(), locale.c_str());
}

void Schema::set_short_desc(const Glib::ustring& desc)
{
  gconf_schema_set_short_desc(gobj(), desc.c_str());
}

void Schema::set_long_desc(const Glib::ustring& desc)
{
  gconf_schema_set_long_desc(gobj(), desc.c_str());
}

void Schema::set_owner(const Glib::ustring& owner)
{
  gconf_schema_set_owner(gobj(), owner.c_str());
}

void Schema::set_default_value(const Value& val)
{
  gconf_schema_set_default_value(gobj(), const_cast<GConfValue*>(val.gobj()));
}

GConfValueType Schema::get_type() const
{
  return gconf_schema_get_type(gobj());
}

GConfValueType Schema::get_list_type() const
{
  return gconf_schema_get_list_type(gobj());
}

GConfValueType Schema::get_car_type() const
{
  return gconf_schema_get_car_type(gobj());
}

GConfValueType Schema::get_cdr_type() const
{
  return gconf_schema_get_cdr_type(gobj());
}

std::string Schema::get_locale() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_schema_get_locale(gobj());
  if(pchVal)
    strResult = pchVal;

  return pchVal;
}

Glib::ustring Schema::get_short_desc() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_schema_get_short_desc(gobj());
  if(pchVal)
    strResult = pchVal;

  return pchVal;
}

Glib::ustring Schema::get_long_desc() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_schema_get_long_desc(gobj());
  if(pchVal)
    strResult = pchVal;

  return pchVal;
}

Glib::ustring Schema::get_owner() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_schema_get_owner(gobj());
  if(pchVal)
    strResult = pchVal;

  return pchVal;
}

Value Schema::get_default_value() const
{
  GConfValue* pValue = gconf_schema_get_default_value(const_cast<GConfSchema*>(gobj()));
  return Value(pValue);
}


} /* namespace Conf */
} /* namespace Gnome */
