/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome-db-entry.h>

struct _GnomeDbEntryPrivate {
	GdaRecordset* recset;
	gint           col;
};

static void gnome_db_entry_class_init (GnomeDbEntryClass *klass);
static void gnome_db_entry_init       (GnomeDbEntry *entry);
static void gnome_db_entry_destroy    (GtkObject *object);

/*
 * Callbacks
 */
static void
row_changed_cb (GdaRecordset *rs, GnomeDbEntry *entry)
{
	GdaField* field;
	
	g_return_if_fail(GDA_IS_RECORDSET(rs));
	g_return_if_fail(GNOME_DB_IS_ENTRY(entry));
	g_return_if_fail(entry->priv->recset == rs);
	
	field = gda_recordset_field_idx(entry->priv->recset, entry->priv->col);
	if (field) {
		gnome_db_entry_set_text(entry, gda_stringify_value(0, 0, field));
	}
}

/*
 * Private functions
 */
static void
gnome_db_entry_destroy (GtkObject *object)
{
	GtkObjectClass *parent_class;
	GnomeDbEntry *entry = (GnomeDbEntry *) object;

	g_return_if_fail(GNOME_DB_IS_ENTRY(entry));
	
	if (entry->priv) {
		if (GDA_IS_RECORDSET(entry->priv->recset))
			gda_recordset_free (entry->priv->recset);
		g_free((gpointer) entry->priv);
		entry->priv = NULL;
	}

	parent_class = gtk_type_class (gtk_entry_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

static void
gnome_db_entry_class_init (GnomeDbEntryClass *klass)
{
	GtkObjectClass *object_class;

	object_class->destroy = gnome_db_entry_destroy;
}

static void
gnome_db_entry_init (GnomeDbEntry *entry)
{
	g_return_if_fail(GNOME_DB_IS_ENTRY(entry));
	
	entry->priv = g_new0(GnomeDbEntryPrivate, 1);
}

/*
 * Public functions
 */
GtkType
gnome_db_entry_get_type (void) {
	static GtkType db_entry_type = 0;
	
	if (!db_entry_type) {
		GtkTypeInfo db_entry_info = {
			"GnomeDbEntry",
			sizeof (GnomeDbEntry),
			sizeof (GnomeDbEntryClass),
			(GtkClassInitFunc) gnome_db_entry_class_init,
			(GtkObjectInitFunc) gnome_db_entry_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_entry_type = gtk_type_unique(gtk_entry_get_type(), &db_entry_info);
	}
	return (db_entry_type);
}

/**
 * gnome_db_entry_new
 */
GtkWidget *
gnome_db_entry_new (GdaRecordset *recset, gint col)
{
	GnomeDbEntry* entry;
	
	entry = GNOME_DB_ENTRY(gtk_type_new(gnome_db_entry_get_type()));
	if (GDA_IS_RECORDSET(recset))
		gnome_db_entry_set_recordset(entry, recset, col);
	return GTK_WIDGET(entry);
}

/**
 * gnome_db_entry_get_recordset
 */
GdaRecordset *
gnome_db_entry_get_recordset (GnomeDbEntry *entry)
{
	g_return_val_if_fail(GNOME_DB_IS_ENTRY(entry), NULL);
	g_return_val_if_fail(entry->priv != NULL, NULL);
	return entry->priv->recset;
}

/**
 * gnome_db_entry_set_recordset
 */
void
gnome_db_entry_set_recordset (GnomeDbEntry *entry, GdaRecordset *recset, gint col)
{
	g_return_if_fail(GNOME_DB_IS_ENTRY(entry));
	g_return_if_fail(entry->priv != NULL);
	
	/* remove the currently active recordset */
	if (GDA_IS_RECORDSET(entry->priv->recset)) {
		gtk_signal_disconnect(GTK_OBJECT(entry->priv->recset),
		                      gtk_signal_lookup("row_changed", gda_recordset_get_type()));
		gda_recordset_free (entry->priv->recset);
	}
	
	/* set the new recordset */
	gtk_object_ref (GTK_OBJECT (recset));
	entry->priv->recset = recset;
	entry->priv->col = col;
	gtk_signal_connect(GTK_OBJECT(entry->priv->recset),
	                   "row_changed",
	                   GTK_SIGNAL_FUNC(row_changed_cb),
	                   (gpointer) entry);
	gtk_object_ref(GTK_OBJECT(entry->priv->recset));
}

/**
 * gnome_db_entry_get_text
 */
const gchar *
gnome_db_entry_get_text (GnomeDbEntry *entry)
{
	g_return_val_if_fail(GNOME_DB_IS_ENTRY(entry), NULL);
	return (const gchar *) gtk_entry_get_text(GTK_ENTRY(entry));
}

/**
 * gnome_db_entry_set_text
 */
void
gnome_db_entry_set_text (GnomeDbEntry *entry, const gchar *txt)
{
}
