/*
 * gnome-keyring
 *
 * Copyright (C) 2008 Stefan Walter
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "config.h"

#include "pkcs11/pkcs11.h"

#include "gck-attributes.h"
#include "gck-crypto.h"
#include "gck-secret-key.h"
#include "gck-session.h"
#include "gck-util.h"

struct _GckSecretKeyPrivate {
	gpointer id;
	gsize n_id;
};

G_DEFINE_TYPE (GckSecretKey, gck_secret_key, GCK_TYPE_OBJECT);

/* -----------------------------------------------------------------------------
 * INTERNAL
 */

/* -----------------------------------------------------------------------------
 * PUBLIC_SECRET_KEY
 */

static CK_RV
gck_secret_key_real_get_attribute (GckObject *base, GckSession *session, CK_ATTRIBUTE* attr)
{
	GckSecretKey *self = GCK_SECRET_KEY (base);

	switch (attr->type)
	{
	case CKA_CLASS:
		return gck_attribute_set_ulong (attr, CKO_SECRET_KEY);

	case CKA_SENSITIVE:
	case CKA_ENCRYPT:
	case CKA_DECRYPT:
	case CKA_SIGN:
	case CKA_VERIFY:
	case CKA_WRAP:
	case CKA_UNWRAP:
	case CKA_DERIVE:
		return gck_attribute_set_bool (attr, FALSE);

	case CKA_EXTRACTABLE:
		return gck_attribute_set_bool (attr, TRUE);

	case CKA_ALWAYS_SENSITIVE:
		return gck_attribute_set_bool (attr, FALSE);

	case CKA_NEVER_EXTRACTABLE:
		return gck_attribute_set_bool (attr, FALSE);

	case CKA_WRAP_WITH_TRUSTED:
		return gck_attribute_set_bool (attr, FALSE);

	case CKA_TRUSTED:
		return gck_attribute_set_bool (attr, FALSE);

	case CKA_WRAP_TEMPLATE:
	case CKA_UNWRAP_TEMPLATE:
		return CKR_ATTRIBUTE_TYPE_INVALID;

	case CKA_START_DATE:
	case CKA_END_DATE:
		return gck_attribute_set_empty (attr);

	case CKA_LOCAL:
		return gck_attribute_set_bool (attr, FALSE);

	case CKA_ID:
		return gck_attribute_set_data (attr, self->pv->id, self->pv->n_id);

	case CKA_KEY_GEN_MECHANISM:
		return gck_attribute_set_ulong (attr, CK_UNAVAILABLE_INFORMATION);
	};

	return GCK_OBJECT_CLASS (gck_secret_key_parent_class)->get_attribute (base, session, attr);
}

static void
gck_secret_key_real_create_attributes (GckObject *object, GckSession *session, GckTransaction *transaction,
                                       CK_ATTRIBUTE *attrs, CK_ULONG n_attrs)
{
	GckSecretKey *self = GCK_SECRET_KEY (object);
	CK_ATTRIBUTE_PTR id;

	if (!self->pv->n_id) {
		id = gck_attributes_find (attrs, n_attrs, CKA_ID);
		if (id == NULL) {
			self->pv->id = NULL;
			self->pv->n_id = 0;
		} else {
			self->pv->id = g_memdup (id->pValue, id->ulValueLen);
			self->pv->n_id = id->ulValueLen;
			gck_attribute_consume (id);
		}
	}
}

static void
gck_secret_key_init (GckSecretKey *self)
{
	self->pv = G_TYPE_INSTANCE_GET_PRIVATE (self, GCK_TYPE_SECRET_KEY, GckSecretKeyPrivate);
}

static void
gck_secret_key_finalize (GObject *obj)
{
	GckSecretKey *self = GCK_SECRET_KEY (obj);

	g_free (self->pv->id);
	self->pv->id = NULL;
	self->pv->n_id = 0;

	G_OBJECT_CLASS (gck_secret_key_parent_class)->finalize (obj);
}

static void
gck_secret_key_class_init (GckSecretKeyClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
	GckObjectClass *gck_class = GCK_OBJECT_CLASS (klass);

	gck_secret_key_parent_class = g_type_class_peek_parent (klass);

	gobject_class->finalize = gck_secret_key_finalize;

	gck_class->get_attribute = gck_secret_key_real_get_attribute;
	gck_class->create_attributes = gck_secret_key_real_create_attributes;

	g_type_class_add_private (klass, sizeof (GckSecretKeyPrivate));
}

/* -----------------------------------------------------------------------------
 * PUBLIC
 */

