<?xml version='1.0' encoding='UTF-8'?><!-- -*- indent-tabs-mode: nil -*- -->
<!--
This program is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
details.

You should have received a copy of the GNU Lesser General Public License
along with this program; see the file COPYING.LGPL.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.
-->

<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:mal="http://projectmallard.org/1.0/"
                xmlns:e="http://projectmallard.org/experimental/"
                xmlns:exsl="http://exslt.org/common"
                xmlns="http://www.w3.org/1999/xhtml"
                exclude-result-prefixes="mal e exsl"
                version="1.0">

<!--!!==========================================================================
Mallard to HTML - Pages

REMARK: Describe this module
-->


<!--@@==========================================================================
mal2html.editor_mode
Add information that's useful to writers and editors.

When this parameter is set to true, these stylesheets will output editorial
comments, status markers, and other information that's useful to writers and
editors.
-->
<xsl:param name="mal2html.editor_mode" select="false()"/>


<!--**==========================================================================
mal2html.page.copyrights
Output the copyright notice at the bottom of a page.
:Revision:version="1.0" date="2010-01-02"
$node: The top-level #{page} element.

This template outputs copyright information.  By default, it is placed at the
bottom of the page by *{mal2html.page.footbar}.  Copyrights are taken from the
#{credit} elements in the #{info} element in ${node}.

Copyright information is output in a #{div} element with #{class="copyrights"}.
Each copyright is output in a nested #{div} element with #{class="copyright"}.
-->
<xsl:template name="mal2html.page.copyrights">
  <xsl:param name="node" select="."/>
  <div class="copyrights">
    <xsl:for-each select="$node/mal:info/mal:credit[mal:years]">
      <div class="copyright">
        <!-- FIXME: i18n, multi-year, email -->
        <xsl:value-of select="concat('© ', mal:years, ' ', mal:name)"/>
      </div>
    </xsl:for-each>
  </div>
</xsl:template>


<!--**==========================================================================
mal2html.page.linkdiv
Output an automatic link block from a guide to a page.
$source: The #{page} or #{section} element containing the link.
$target: The element from the cache file of the page being linked to.
$class: An additional string to prepend to the #{class} attribute.
$attrs: A set of extra data attributes to add to the #{a} element.

REMARK: Describe this template
-->
<xsl:template name="mal2html.page.linkdiv">
  <xsl:param name="source" select="."/>
  <xsl:param name="target"/>
  <xsl:param name="class" select="''"/>
  <xsl:param name="attrs"/>
  <a class="{concat($class, ' linkdiv')}">
    <xsl:attribute name="href">
      <xsl:call-template name="mal.link.target">
        <xsl:with-param name="node" select="$source"/>
        <xsl:with-param name="xref" select="$target/@id"/>
      </xsl:call-template>
    </xsl:attribute>
    <xsl:copy-of select="exsl:node-set($attrs)/*/@*"/>
    <div class="linkdiv">
      <div class="title">
        <span class="title">
          <xsl:call-template name="mal.link.content">
            <xsl:with-param name="node" select="$source"/>
            <xsl:with-param name="xref" select="$target/@id"/>
            <xsl:with-param name="role" select="'topic'"/>
          </xsl:call-template>
        </span>
        <xsl:call-template name="mal2html.editor.badge">
          <xsl:with-param name="target" select="$target"/>
        </xsl:call-template>
      </div>
      <xsl:if test="$target/mal:info/mal:desc">
        <div class="desc">
          <xsl:apply-templates mode="mal2html.inline.mode"
                               select="$target/mal:info/mal:desc[1]/node()"/>
        </div>
      </xsl:if>
    </div>
  </a>
</xsl:template>


<!--**==========================================================================
mal2html.page.autolinks
Outputs the automatic links from a page to related pages
$node: The #{topic} or #{section} element containing the links

REMARK: Describe this template
-->
<xsl:template name="mal2html.page.autolinks">
  <xsl:param name="node" select="."/>
  <xsl:variable name="depth"
                select="count($node/ancestor::mal:section) + 2"/>
  <xsl:variable name="id">
    <xsl:choose>
      <xsl:when test="$node/self::mal:section">
        <xsl:value-of select="concat($node/ancestor::mal:page[1]/@id, '#', $node/@id)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$node/@id"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="guidelinks">
    <xsl:call-template name="mal.link.guidelinks">
      <xsl:with-param name="node" select="$node"/>
    </xsl:call-template>
  </xsl:variable>
  <xsl:variable name="guidenodes" select="exsl:node-set($guidelinks)/*"/>
  <xsl:variable name="seealsolinks">
    <xsl:call-template name="mal.link.seealsolinks">
      <xsl:with-param name="node" select="$node"/>
    </xsl:call-template>
  </xsl:variable>
  <xsl:variable name="seealsonodes" select="exsl:node-set($seealsolinks)/*"/>
  <xsl:if test="$guidenodes or $seealsonodes">
    <div class="sect sect-links">
      <div class="hgroup">
        <xsl:element name="{concat('h', $depth)}" namespace="{$html.namespace}">
          <xsl:call-template name="l10n.gettext">
            <xsl:with-param name="msgid" select="'Further Reading'"/>
          </xsl:call-template>
        </xsl:element>
      </div>
      <!-- FIXME: For prev/next series, insert links to first/prev/next/last -->
      <div class="links">
        <xsl:if test="$guidenodes">
          <div class="title"><span class="title">
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'More About'"/>
            </xsl:call-template>
          </span></div>
          <ul>
            <xsl:for-each select="$guidenodes">
              <xsl:sort select="mal:title[@type = 'sort']"/>
              <xsl:call-template name="mal2html.page.autolink">
                <xsl:with-param name="xref" select="@xref"/>
                <xsl:with-param name="role" select="'guide'"/>
              </xsl:call-template>
            </xsl:for-each>
          </ul>
        </xsl:if>
        <xsl:if test="$seealsonodes">
          <div class="title"><span class="title">
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'See Also'"/>
            </xsl:call-template>
          </span></div>
          <ul>
            <xsl:for-each select="$seealsonodes">
              <xsl:sort select="mal:title[@type = 'sort']"/>
              <xsl:call-template name="mal2html.page.autolink">
                <xsl:with-param name="xref" select="@xref"/>
                <xsl:with-param name="role" select="'seealso'"/>
              </xsl:call-template>
            </xsl:for-each>
          </ul>
        </xsl:if>
      </div>
    </div>
  </xsl:if>
</xsl:template>


<!--**==========================================================================
mal2html.page.autolink
Outputs an automatic link for a related page
$page: The element from the cache file of the page being linked to

REMARK: Describe this template
-->
<xsl:template name="mal2html.page.autolink">
  <xsl:param name="page"/>
  <xsl:param name="xref" select="$page/@id"/>
  <xsl:param name="role" select="''"/>
  <xsl:for-each select="$mal.cache">
    <xsl:variable name="target" select="key('mal.cache.key', $xref)"/>
    <li class="links">
      <a>
        <xsl:attribute name="href">
          <xsl:call-template name="mal.link.target">
            <xsl:with-param name="xref" select="$xref"/>
          </xsl:call-template>
        </xsl:attribute>
        <xsl:call-template name="mal.link.content">
          <xsl:with-param name="node" select="."/>
          <xsl:with-param name="xref" select="$xref"/>
          <xsl:with-param name="role" select="$role"/>
        </xsl:call-template>
      </a>
      <xsl:if test="$role = 'guide'">
        <xsl:call-template name="mal2html.editor.badge">
          <xsl:with-param name="target" select="$target"/>
        </xsl:call-template>
      </xsl:if>
      <xsl:variable name="desc" select="$target/mal:info/mal:desc"/>
      <xsl:if test="$desc">
        <span class="desc">
          <xsl:text> &#x2014; </xsl:text>
          <xsl:apply-templates mode="mal2html.inline.mode" select="$desc/node()"/>
        </span>
      </xsl:if>
    </li>
  </xsl:for-each>
</xsl:template>





<!--**==========================================================================
mal2html.page.linktrails
FIXME

REMARK: Describe this template
-->
<xsl:template name="mal2html.page.linktrails">
  <xsl:param name="node" select="."/>
  <xsl:variable name="linktrails">
    <xsl:call-template name="mal.link.linktrails">
      <xsl:with-param name="node" select="$node"/>
    </xsl:call-template>
  </xsl:variable>
  <xsl:variable name="trailnodes" select="exsl:node-set($linktrails)/*"/>
  <xsl:if test="count($trailnodes) &gt; 0">
    <div class="trails">
      <xsl:for-each select="$trailnodes">
        <xsl:sort select="(.//mal:title[@type='sort'])[1]"/>
        <xsl:sort select="(.//mal:title[@type='sort'])[2]"/>
        <xsl:sort select="(.//mal:title[@type='sort'])[3]"/>
        <xsl:sort select="(.//mal:title[@type='sort'])[4]"/>
        <xsl:sort select="(.//mal:title[@type='sort'])[5]"/>
        <xsl:call-template name="mal2html.page.linktrails.trail"/>
      </xsl:for-each>
    </div>
  </xsl:if>
</xsl:template>

<!--**==========================================================================
mal2html.page.linktrails.trail
FIXME

REMARK: Describe this template
-->
<xsl:template name="mal2html.page.linktrails.trail">
  <xsl:param name="node" select="."/>
  <div class="trail">
    <xsl:call-template name="mal2html.page.linktrails.link">
      <xsl:with-param name="node" select="$node"/>
    </xsl:call-template>
  </div>
</xsl:template>

<!--**==========================================================================
mal2html.page.linktrails.link
FIXME

REMARK: Describe this template
-->
<xsl:template name="mal2html.page.linktrails.link">
  <xsl:param name="node" select="."/>
  <a class="trail">
    <xsl:attribute name="href">
      <xsl:call-template name="mal.link.target">
        <xsl:with-param name="xref" select="$node/@xref"/>
      </xsl:call-template>
    </xsl:attribute>
    <xsl:call-template name="mal.link.content">
      <xsl:with-param name="node" select="$node"/>
      <xsl:with-param name="xref" select="$node/@xref"/>
      <xsl:with-param name="role" select="'guide'"/>
    </xsl:call-template>
  </a>
  <xsl:choose>
    <xsl:when test="$node/@child = 'section'">
      <xsl:text>&#x00A0;› </xsl:text>
    </xsl:when>
    <xsl:otherwise>
      <xsl:text>&#x00A0;» </xsl:text>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:for-each select="$node/mal:link">
    <xsl:call-template name="mal2html.page.linktrails.link"/>
  </xsl:for-each>
</xsl:template>

<!-- links -->
<xsl:template name="mal2html.links.next" match="e:links[@type = 'next']">
  <xsl:param name="node" select="./self::e:links/.. | ./self::mal:page"/>
  <xsl:variable name="linkid">
    <xsl:call-template name="mal.link.linkid">
      <xsl:with-param name="node" select="$node"/>
    </xsl:call-template>
  </xsl:variable>
  <xsl:variable name="next" select="$node/mal:info/mal:link[@type='next']"/>
  <xsl:for-each select="$mal.cache">
    <xsl:variable name="prev" select="key('mal.cache.link.key', concat('next:', $linkid))"/>
    <xsl:if test="$prev or $next">
      <!-- FIXME: Get prev/next links in constant position -->
      <div class="links nextlinks">
        <xsl:if test="$prev">
          <a class="nextlinks-prev">
            <xsl:attribute name="href">
              <xsl:call-template name="mal.link.target">
                <xsl:with-param name="node" select="$prev"/>
                <xsl:with-param name="xref" select="$prev/../../@id"/>
              </xsl:call-template>
            </xsl:attribute>
            <xsl:attribute name="title">
              <xsl:call-template name="mal.link.content">
                <xsl:with-param name="node" select="$prev"/>
                <xsl:with-param name="xref" select="$prev/../../@id"/>
              </xsl:call-template>
            </xsl:attribute>
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'Previous'"/>
            </xsl:call-template>
          </a>
        </xsl:if>
        <xsl:if test="$prev and $next">
          <xsl:text>&#x00A0;&#x00A0;|&#x00A0;&#x00A0;</xsl:text>
        </xsl:if>
        <xsl:if test="$next">
          <a class="nextlinks-next">
            <xsl:attribute name="href">
              <xsl:call-template name="mal.link.target">
                <xsl:with-param name="node" select="$next"/>
                <xsl:with-param name="xref" select="$next/@xref"/>
              </xsl:call-template>
            </xsl:attribute>
            <xsl:attribute name="title">
              <xsl:call-template name="mal.link.content">
                <xsl:with-param name="node" select="$next"/>
                <xsl:with-param name="xref" select="$next/@xref"/>
              </xsl:call-template>
            </xsl:attribute>
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'Next'"/>
            </xsl:call-template>
          </a>
        </xsl:if>
      </div>
    </xsl:if>
  </xsl:for-each>
</xsl:template>

<xsl:template name="mal2html.links.series.prev">
  <xsl:param name="node" select="."/>
  <xsl:variable name="linkid">
    <xsl:call-template name="mal.link.linkid">
      <xsl:with-param name="node" select="$node"/>
    </xsl:call-template>
  </xsl:variable>
  <xsl:for-each select="$mal.cache">
    <xsl:variable name="prev" select="key('mal.cache.link.key', concat('next:', $linkid))"/>
    <xsl:if test="$prev">
      <xsl:call-template name="mal2html.links.series.prev">
        <xsl:with-param name="node" select="key('mal.cache.key', $prev/../../@id)"/>
      </xsl:call-template>
      <li class="links">
        <a>
          <xsl:attribute name="href">
            <xsl:call-template name="mal.link.target">
              <xsl:with-param name="node" select="$prev"/>
              <xsl:with-param name="xref" select="$prev/../../@id"/>
            </xsl:call-template>
          </xsl:attribute>
          <xsl:call-template name="mal.link.content">
            <xsl:with-param name="node" select="$prev"/>
            <xsl:with-param name="xref" select="$prev/../../@id"/>
          </xsl:call-template>
        </a>
      </li>
    </xsl:if>
  </xsl:for-each>
</xsl:template>

<xsl:template name="mal2html.links.series.next">
  <xsl:param name="node" select="."/>
  <xsl:variable name="linkid">
    <xsl:call-template name="mal.link.linkid">
      <xsl:with-param name="node" select="$node"/>
    </xsl:call-template>
  </xsl:variable>
  <xsl:variable name="next" select="$node/mal:info/mal:link[@type='next']"/>
  <xsl:if test="$next">
    <xsl:for-each select="$mal.cache">
      <li class="links">
        <a>
          <xsl:attribute name="href">
            <xsl:call-template name="mal.link.target">
              <xsl:with-param name="node" select="$next"/>
              <xsl:with-param name="xref" select="$next/@xref"/>
            </xsl:call-template>
          </xsl:attribute>
          <xsl:call-template name="mal.link.content">
            <xsl:with-param name="node" select="$next"/>
            <xsl:with-param name="xref" select="$next/@xref"/>
          </xsl:call-template>
        </a>
      </li>
      <xsl:call-template name="mal2html.links.series.next">
        <xsl:with-param name="node" select="key('mal.cache.key', $next/@xref)"/>
      </xsl:call-template>
    </xsl:for-each>
  </xsl:if>
</xsl:template>

<xsl:template match="e:links[@type = 'series']">
  <div class="links serieslinks">
    <xsl:apply-templates mode="mal2html.block.mode" select="mal:title"/>
    <ul>
      <xsl:call-template name="mal2html.links.series.prev">
        <xsl:with-param name="node" select="/mal:page"/>
      </xsl:call-template>
      <li class="links">
        <xsl:call-template name="mal.link.content">
          <xsl:with-param name="node" select="/mal:page"/>
          <xsl:with-param name="xref" select="/mal:page/@id"/>
        </xsl:call-template>
      </li>
      <xsl:call-template name="mal2html.links.series.next">
        <xsl:with-param name="node" select="/mal:page"/>
      </xsl:call-template>
    </ul>
  </div>
</xsl:template>

<xsl:template name="mal2html.editor.badge">
  <xsl:param name="target" select="."/>
  <xsl:if test="$mal2html.editor_mode">
    <xsl:variable name="page" select="$target/ancestor-or-self::mal:page[1]"/>
    <xsl:variable name="date">
      <xsl:for-each select="$page/mal:info/mal:revision">
        <xsl:sort select="@date" data-type="text" order="descending"/>
        <xsl:if test="position() = 1">
          <xsl:value-of select="@date"/>
        </xsl:if>
      </xsl:for-each>
    </xsl:variable>
    <xsl:variable name="revision"
                  select="$page/mal:info/mal:revision[@date = $date][last()]"/>
    <xsl:if test="$revision/@status != '' and $revision/@status != 'final'">
      <xsl:text> </xsl:text>
      <span>
        <xsl:attribute name="class">
          <xsl:value-of select="concat('status status-', $revision/@status)"/>
        </xsl:attribute>
        <!-- FIXME: i18n -->
        <xsl:choose>
          <xsl:when test="$revision/@status = 'stub'">
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'Stub'"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:when test="$revision/@status = 'incomplete'">
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'Incomplete'"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:when test="$revision/@status = 'draft'">
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'Draft'"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:when test="$revision/@status = 'review'">
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'Ready for review'"/>
            </xsl:call-template>
          </xsl:when>
          <xsl:when test="$revision/@status = 'final'">
            <xsl:call-template name="l10n.gettext">
              <xsl:with-param name="msgid" select="'Final'"/>
            </xsl:call-template>
          </xsl:when>
        </xsl:choose>
      </span>
    </xsl:if>
  </xsl:if>
</xsl:template>

<xsl:template name="mal2html.editor.banner">
  <xsl:param name="node" select="."/>
  <xsl:if test="$mal2html.editor_mode">
    <xsl:variable name="date">
      <xsl:for-each select="$node/mal:info/mal:revision">
        <xsl:sort select="@date" data-type="text" order="descending"/>
        <xsl:if test="position() = 1">
          <xsl:value-of select="@date"/>
        </xsl:if>
      </xsl:for-each>
    </xsl:variable>
    <xsl:variable name="revision"
                  select="$node/mal:info/mal:revision[@date = $date][last()]"/>
    <xsl:if test="$revision/@status != ''">
      <div class="version">
        <!-- FIXME: i18n -->
        <div class="title">
          <xsl:choose>
            <xsl:when test="$revision/@status = 'stub'">
              <xsl:call-template name="l10n.gettext">
                <xsl:with-param name="msgid" select="'Stub'"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:when test="$revision/@status = 'incomplete'">
              <xsl:call-template name="l10n.gettext">
                <xsl:with-param name="msgid" select="'Incomplete'"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:when test="$revision/@status = 'draft'">
              <xsl:call-template name="l10n.gettext">
                <xsl:with-param name="msgid" select="'Draft'"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:when test="$revision/@status = 'review'">
              <xsl:call-template name="l10n.gettext">
                <xsl:with-param name="msgid" select="'Ready for review'"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:when test="$revision/@status = 'final'">
              <xsl:call-template name="l10n.gettext">
                <xsl:with-param name="msgid" select="'Final'"/>
              </xsl:call-template>
            </xsl:when>
          </xsl:choose>
        </div>
        <p class="version">
          <!-- FIXME: i18n -->
          <xsl:text>Version </xsl:text>
          <xsl:value-of select="$revision/@version"/>
          <xsl:text> on </xsl:text>
          <xsl:value-of select="$revision/@date"/>
        </p>
        <xsl:apply-templates mode="mal2html.block.mode" select="$revision/*"/>
      </div>
    </xsl:if>
  </xsl:if>
</xsl:template>

<!-- == Matched Templates == -->


<xsl:template mode="html.title.mode" match="mal:page">
  <xsl:variable name="title" select="mal:info/mal:title[@type = 'text'][1]"/>
  <xsl:choose>
    <xsl:when test="$title">
      <xsl:value-of select="$title"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="mal:title"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template mode="html.header.mode" match="mal:page">
  <xsl:call-template name="mal2html.page.linktrails"/>
</xsl:template>

<xsl:template mode="html.footer.mode" match="mal:page">
  <xsl:call-template name="mal2html.page.copyrights"/>
</xsl:template>

<xsl:template mode="html.body.mode" match="mal:page">
  <xsl:call-template name="mal2html.editor.banner"/>
  <xsl:choose>
    <xsl:when test="not(e:links[@type = 'next'])">
      <xsl:call-template name="mal2html.links.next"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates
          select="e:links[@type = 'next'][contains(concat(' ', @style, ' '), ' top ')]">
      </xsl:apply-templates>
    </xsl:otherwise>
  </xsl:choose>
  <xsl:apply-templates select="."/>
</xsl:template>

<!-- = section = -->
<xsl:template mode="mal2html.section.mode" match="mal:section">
  <div class="sect" id="{@id}">
    <xsl:apply-templates select="."/>
  </div>
</xsl:template>

<!-- page | section -->
<xsl:template match="mal:page | mal:section">
  <xsl:variable name="type" select="/mal:page/@type"/>
  <xsl:variable name="topiclinks">
    <xsl:if test="$type = 'guide'">
      <xsl:call-template name="mal.link.topiclinks"/>
    </xsl:if>
  </xsl:variable>
  <xsl:variable name="topicnodes" select="exsl:node-set($topiclinks)/*"/>
  <xsl:variable name="allgroups">
    <xsl:if test="$type = 'guide'">
      <xsl:text> </xsl:text>
      <xsl:for-each select="e:links[@type = 'topic']">
        <xsl:choose>
          <xsl:when test="@groups">
            <xsl:value-of select="@groups"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>#default</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:text> </xsl:text>
      </xsl:for-each>
    </xsl:if>
  </xsl:variable>
  <div class="hgroup">
    <xsl:apply-templates mode="mal2html.title.mode" select="mal:title"/>
    <xsl:apply-templates mode="mal2html.title.mode" select="mal:subtitle"/>
  </div>
  <div class="contents">
    <xsl:if test="$type = 'facets'">
      <xsl:call-template name="mal2html.facets.controls"/>
    </xsl:if>
    <xsl:for-each
        select="*[not(self::mal:section or self::mal:title or self::mal:subtitle)]">
      <xsl:choose>
        <xsl:when test="preceding-sibling::mal:section"/>
        <xsl:when test="self::e:links[@type = 'topic']">
          <xsl:if test="$type = 'guide'">
            <xsl:apply-templates select=".">
              <xsl:with-param name="allgroups" select="$allgroups"/>
              <xsl:with-param name="links" select="$topicnodes"/>
            </xsl:apply-templates>
          </xsl:if>
        </xsl:when>
        <xsl:when test="self::e:links[@type = 'next']">
          <xsl:if test="not(contains(concat(' ', @style, ' '), ' top '))">
            <xsl:apply-templates select="."/>
          </xsl:if>
        </xsl:when>
        <xsl:when test="self::e:links">
          <xsl:apply-templates select="."/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates mode="mal2html.block.mode" select="."/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:for-each>
    <xsl:if test="$type = 'guide'">
      <xsl:if test="not(e:links[@type = 'topic'])">
        <xsl:call-template name="mal2html.links.topic">
          <xsl:with-param name="links" select="$topicnodes"/>
        </xsl:call-template>
      </xsl:if>
    </xsl:if>
    <xsl:if test="$type = 'facets'">
      <xsl:call-template name="mal2html.facets.links"/>
    </xsl:if>
  </div>
  <xsl:apply-templates mode="mal2html.section.mode" select="mal:section"/>
  <xsl:call-template name="mal2html.page.autolinks">
    <xsl:with-param name="node" select="."/>
  </xsl:call-template>
</xsl:template>

<!-- links -->
<xsl:template name="mal2html.links.topic" match="e:links[@type = 'topic']">
  <xsl:param name="node" select="."/>
  <xsl:param name="groups">
    <xsl:text> </xsl:text>
    <xsl:choose>
      <xsl:when test="$node/@groups">
        <xsl:value-of select="$node/@groups"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>#default</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:text> </xsl:text>
  </xsl:param>
  <xsl:param name="allgroups" select="''"/>
  <xsl:param name="links" select="/false"/>
  <xsl:if test="/mal:page/@type = 'guide'">
    <xsl:variable name="_groups">
      <xsl:if test="not(contains($allgroups, ' #first '))">
        <xsl:if test="not($node/self::e:links) or not($node/preceding-sibling::e:links[@type = 'topic'])">
          <xsl:text> #first </xsl:text>
        </xsl:if>
      </xsl:if>
      <xsl:value-of select="$groups"/>
      <xsl:if test="not(contains($allgroups, ' #default '))">
        <xsl:if test="not($node/self::e:links) or not($node/following-sibling::e:links[@type = 'topic'])">
          <xsl:text> #default </xsl:text>
        </xsl:if>
      </xsl:if>
      <xsl:if test="not(contains($allgroups, ' #last '))">
        <xsl:if test="not($node/self::e:links) or not($node/following-sibling::e:links[@type = 'topic'])">
          <xsl:text> #last </xsl:text>
        </xsl:if>
      </xsl:if>
    </xsl:variable>
    <xsl:variable name="_links" select="$links[contains($_groups, concat(' ', @group, ' '))]"/>
    <xsl:if test="count($_links) != 0">
      <div class="links topiclinks">
        <xsl:if test="$node/self::e:links">
          <xsl:apply-templates mode="mal2html.block.mode" select="$node/mal:title"/>
        </xsl:if>
        <xsl:choose>
          <xsl:when test="contains(concat(' ', $node/@style, ' '), ' linklist ')">
            <ul>
              <xsl:for-each select="$_links">
                <xsl:sort data-type="number" select="@groupsort"/>
                <xsl:sort select="mal:title[@type = 'sort']"/>
                <xsl:call-template name="mal2html.page.autolink">
                  <xsl:with-param name="xref" select="@xref"/>
                  <xsl:with-param name="role" select="'guide'"/>
                </xsl:call-template>
              </xsl:for-each>
            </ul>
          </xsl:when>
          <xsl:when test="contains(concat(' ', $node/@style, ' '), ' 2column ')">
            <xsl:variable name="coltot" select="ceiling(count($_links) div 2)"/>
            <table class="twocolumn"><tr>
              <td class="twocolumnleft">
                <xsl:for-each select="$_links">
                  <xsl:sort data-type="number" select="@groupsort"/>
                  <xsl:sort select="mal:title[@type = 'sort']"/>
                  <xsl:if test="position() &lt;= $coltot">
                    <xsl:variable name="xref" select="@xref"/>
                    <xsl:for-each select="$mal.cache">
                      <xsl:call-template name="mal2html.page.linkdiv">
                        <xsl:with-param name="source" select="$node"/>
                        <xsl:with-param name="target" select="key('mal.cache.key', $xref)"/>
                      </xsl:call-template>
                    </xsl:for-each>
                  </xsl:if>
                </xsl:for-each>
              </td>
              <td class="twocolumnright">
                <xsl:for-each select="$_links">
                  <xsl:sort data-type="number" select="@groupsort"/>
                  <xsl:sort select="mal:title[@type = 'sort']"/>
                  <xsl:if test="position() &gt; $coltot">
                    <xsl:variable name="xref" select="@xref"/>
                    <xsl:for-each select="$mal.cache">
                      <xsl:call-template name="mal2html.page.linkdiv">
                        <xsl:with-param name="source" select="$node"/>
                        <xsl:with-param name="target" select="key('mal.cache.key', $xref)"/>
                      </xsl:call-template>
                    </xsl:for-each>
                  </xsl:if>
                </xsl:for-each>
              </td>
            </tr></table>
          </xsl:when>
          <xsl:otherwise>
            <xsl:for-each select="$_links">
              <xsl:sort data-type="number" select="@groupsort"/>
              <xsl:sort select="mal:title[@type = 'sort']"/>
              <xsl:variable name="xref" select="@xref"/>
              <xsl:for-each select="$mal.cache">
                <xsl:call-template name="mal2html.page.linkdiv">
                  <xsl:with-param name="source" select="$node"/>
                  <xsl:with-param name="target" select="key('mal.cache.key', $xref)"/>
                </xsl:call-template>
              </xsl:for-each>
            </xsl:for-each>
          </xsl:otherwise>
        </xsl:choose>
      </div>
    </xsl:if>
  </xsl:if>
</xsl:template>

<xsl:template match="e:links[@type = 'section']">
  <xsl:if test="../mal:section">
    <div class="links sectionlinks">
      <xsl:apply-templates mode="mal2html.block.mode" select="mal:title"/>
      <ul>
        <xsl:for-each select="../mal:section">
          <xsl:call-template name="mal2html.page.autolink">
            <xsl:with-param name="xref" select="concat(/mal:page/@id, '#', @id)"/>
            <xsl:with-param name="role" select="'section'"/>
          </xsl:call-template>
        </xsl:for-each>
      </ul>
    </div>
  </xsl:if>
</xsl:template>


<!--%%==========================================================================
mal2html.title.mode
FIXME

FIXME
-->
<!-- = subtitle = -->
<xsl:template mode="mal2html.title.mode" match="mal:subtitle">
  <xsl:variable name="depth"
                select="count(ancestor::mal:section) + 2"/>
  <xsl:element name="{concat('h', $depth)}" namespace="{$html.namespace}">
    <xsl:attribute name="class">
      <xsl:text>title</xsl:text>
    </xsl:attribute>
    <xsl:apply-templates mode="mal2html.inline.mode"/>
  </xsl:element>
</xsl:template>

<!-- = title = -->
<xsl:template mode="mal2html.title.mode" match="mal:title">
  <xsl:variable name="depth"
                select="count(ancestor::mal:section) + 1"/>
  <xsl:element name="{concat('h', $depth)}" namespace="{$html.namespace}">
    <xsl:attribute name="class">
      <xsl:text>title</xsl:text>
    </xsl:attribute>
    <xsl:apply-templates mode="mal2html.inline.mode"/>
  </xsl:element>
</xsl:template>

<!--%# html.css.mode -->
<xsl:template mode="html.css.mode" match="mal:page">
  <xsl:param name="direction">
    <xsl:call-template name="l10n.direction"/>
  </xsl:param>
  <xsl:param name="left">
    <xsl:call-template name="l10n.align.start">
      <xsl:with-param name="direction" select="$direction"/>
    </xsl:call-template>
  </xsl:param>
  <xsl:param name="right">
    <xsl:call-template name="l10n.align.end">
      <xsl:with-param name="direction" select="$direction"/>
    </xsl:call-template>
  </xsl:param>
<xsl:text>
div.floatleft {
  float: left;
  margin-top: 0;
  margin-right: 1em;
}
div.floatright {
  float: right;
  margin-top: 0;
  margin-left: 1em;
}

div.copyrights {
  text-align: center;
  color: </xsl:text>
    <xsl:value-of select="$color.text_light"/><xsl:text>;
}

table.twocolumn { width: 100%; }
td.twocolumnleft { width: 48%; vertical-align: top; padding: 0; margin: 0; }
td.twocolumnright {
  width: 52%; vertical-align: top;
  margin: 0; padding: 0;
  padding-</xsl:text><xsl:value-of select="$left"/><xsl:text>: 1em;
  -webkit-padding-start: 1em;
  -webkit-padding-end: 0;
  -moz-padding-start: 1em;
  -moz-padding-end: 0;
}

a.linkdiv { display: block; }
div.linkdiv div.title {
  font-size: 1em;
  color: inherit;
}
div.linkdiv div.desc {
  color: </xsl:text><xsl:value-of select="$color.text_light"/><xsl:text>;
}
div.linkdiv {
  margin: 0;
  padding: 0.5em;
  -moz-border-radius: 6px;
  border: solid 1px </xsl:text>
    <xsl:value-of select="$color.background"/><xsl:text>;
}
a:hover div.linkdiv {
  text-decoration: none;
  border-color: </xsl:text>
    <xsl:value-of select="$color.blue_border"/><xsl:text>;
  background-color: </xsl:text>
    <xsl:value-of select="$color.blue_background"/><xsl:text>;
}

div.example {
  border-</xsl:text><xsl:value-of select="$left"/><xsl:text>: solid 4px </xsl:text>
    <xsl:value-of select="$color.gray_border"/><xsl:text>;
  padding-</xsl:text><xsl:value-of select="$left"/><xsl:text>: 1em;
}

div.comment {
  padding: 0.5em;
  border: solid 2px </xsl:text>
    <xsl:value-of select="$color.red_border"/><xsl:text>;
  background-color: </xsl:text>
    <xsl:value-of select="$color.red_background"/><xsl:text>;
}
div.comment div.comment {
  margin: 1em 1em 0 1em;
}
div.comment div.cite {
  margin: 0 0 0.5em 0;
  font-style: italic;
}

ul.tree {
  margin: 0; padding: 0;
  list-style-type: none;
}
li.tree { margin: 0; padding: 0; }
li.tree div { margin: 0; padding: 0; }
ul.tree ul.tree {
  margin-</xsl:text><xsl:value-of select="$left"/><xsl:text>: 1.44em;
  -webkit-margin-start: 1.44em;
  -webkit-margin-end: 0;
  -moz-margin-start: 1.44em;
  -moz-margin-end: 0;
}
div.tree-lines ul.tree { margin-left: 0; }

span.hi {
  background-color: </xsl:text>
    <xsl:value-of select="$color.yellow_background"/><xsl:text>;
}

div.facets {
  display: inline-block;
  padding: 6px;
  background-color: </xsl:text>
    <xsl:value-of select="$color.yellow_background"/><xsl:text>;
  border: solid 1px </xsl:text>
    <xsl:value-of select="$color.blue_border"/><xsl:text>;
} 
div.facet {
 vertical-align: top;
  display: inline-block;
  margin-top: 0;
  margin-bottom: 1em;
  margin-</xsl:text><xsl:value-of select="$right"/><xsl:text>: 1em;
}
div.facet div.title { margin: 0; }
div.facet li {
  margin: 0; padding: 0;
  list-style-type: none;
}
div.facet input {
  vertical-align: middle;
  margin: 0;
}
</xsl:text>
<xsl:if test="$mal2html.editor_mode">
<xsl:text>
div.version {
  position: absolute;
  </xsl:text><xsl:value-of select="$right"/><xsl:text>: 12px;
  opacity: 0.2;
  margin-top: -1em;
  padding: 0.5em 1em 0.5em 1em;
  max-width: 24em;
  border: solid 1px </xsl:text>
    <xsl:value-of select="$color.gray_border"/><xsl:text>;
  background-color: </xsl:text>
    <xsl:value-of select="$color.yellow_background"/><xsl:text>;
}
div.version:hover { opacity: 0.8; }
div.version p.version { margin-top: 0.2em; }
span.status {
  font-size: 0.83em;
  font-weight: normal;
  padding-left: 0.2em;
  padding-right: 0.2em;
  color: </xsl:text>
    <xsl:value-of select="$color.text_light"/><xsl:text>;
  border: solid 1px </xsl:text>
    <xsl:value-of select="$color.red_border"/><xsl:text>;
}
span.status-stub { background-color: </xsl:text>
  <xsl:value-of select="$color.red_background"/><xsl:text>; }
span.status-draft { background-color: </xsl:text>
  <xsl:value-of select="$color.red_background"/><xsl:text>; }
span.status-incomplete { background-color: </xsl:text>
  <xsl:value-of select="$color.red_background"/><xsl:text>; }
span.status-review { background-color: </xsl:text>
  <xsl:value-of select="$color.yellow_background"/><xsl:text>; }
</xsl:text>
</xsl:if>
</xsl:template>

<!--%# html.js.mode -->
<xsl:template mode="html.js.mode" match="mal:page">
  <xsl:call-template name="mal2html.facets.js"/>
</xsl:template>

</xsl:stylesheet>
