/*
 * Copyright (C) 2025 Linux Studio Plugins Project <https://lsp-plug.in/>
 *           (C) 2025 Vladimir Sadovnikov <sadko4u@gmail.com>
 *
 * This file is part of lsp-tk-lib
 * Created on: 7 июл. 2017 г.
 *
 * lsp-tk-lib is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * lsp-tk-lib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with lsp-tk-lib. If not, see <https://www.gnu.org/licenses/>.
 */

#include <lsp-plug.in/tk/tk.h>
#include <lsp-plug.in/stdlib/math.h>
#include <private/tk/style/BuiltinStyle.h>

namespace lsp
{
    namespace tk
    {
        namespace style
        {
            LSP_TK_STYLE_IMPL_BEGIN(Indicator, Widget)
                // Bind
                IndicatorColors *c = &vColors[INDICATOR_NORMAL];
                c->sColor.bind("color", this);
                c->sTextColor.bind("text.color", this);

                c = &vColors[INDICATOR_INACTIVE];
                c->sColor.bind("inactive.color", this);
                c->sTextColor.bind("inactive.text.color", this);

                sRows.bind("rows", this);
                sColumns.bind("columns", this);
                sShift.bind("text.shift", this);
                sTextGap.bind("text.gap", this);
                sLoop.bind("text.loop", this);
                sDarkText.bind("text.dark", this);
                sType.bind("type", this);
                sActive.bind("active", this);
                sFont.bind("font", this);
                sSpacing.bind("spacing", this);
                sIPadding.bind("ipadding", this);

                // Configure
                c = &vColors[INDICATOR_NORMAL];
                c->sColor.set("#111111");
                c->sTextColor.set("#00ff00");

                c = &vColors[INDICATOR_INACTIVE];
                c->sColor.set("#111111");
                c->sTextColor.set("#cccccc");

                sRows.set(1);
                sColumns.set(5);
                sShift.set(0);
                sTextGap.set(0);
                sLoop.set(false);
                sDarkText.set(true);
                sType.set(INDICATOR_SEGMENT);
                sActive.set(true);
                sFont.set_size(16);
                sFont.set_bold(true);
                sSpacing.set(0);
                sIPadding.set(1);

                // Override
                sFont.override();
                sSpacing.override();
            LSP_TK_STYLE_IMPL_END
            LSP_TK_BUILTIN_STYLE(Indicator, "Indicator", "root");

            void IndicatorColors::listener(tk::prop::Listener *listener)
            {
                sColor.listener(listener);
                sTextColor.listener(listener);
            }

            bool IndicatorColors::property_changed(Property *prop)
            {
                return prop->one_of(sColor, sTextColor);
            }
        }

        typedef struct point_t
        {
            uint8_t x;
            uint8_t y;
        } point_t;

        typedef struct shape_t
        {
            uint8_t n;
            point_t p[6];
        } shape_t;

        typedef struct segment_t
        {
            uint8_t shape;
            uint8_t dx;
            uint8_t dy;
        } segment_t;

        /**
         * Indicator segments:
         *
         *
         *      33333
         *     2     4
         *     2     4  9
         *     2  0  4
         *      77777
         *     1  0  5  A
         *     1     5
         *     1     5
         *      66666   8
         *
         */

        static const shape_t shapes[] =
        {
            { 4, {  {0, 1}, {1, 0}, {2, 1}, {2, 5}, {1, 6}, {0, 5} }  },  // vertical
            { 4, {  {0, 1}, {1, 0}, {5, 0}, {6, 1}, {5, 2}, {1, 2} }  },  // horizontal
            { 2, {  {0, 0}, {2, 0}, {2, 2}, {0, 2}, {0, 0}, {0, 0} }  },  // dot
        };

        static const segment_t segments[] =
        {
            { 0,  3,  6 },  // 0
            { 0,  0, 10 },  // 1
            { 0,  0,  2 },  // 2
            { 1,  2,  0 },  // 3
            { 0,  8,  2 },  // 4
            { 0,  8, 10 },  // 5
            { 1,  2, 16 },  // 6
            { 1,  2,  8 },  // 7
            { 2, 12, 16 },  // 8
            { 2, 12,  5 },  // 9
            { 2, 12, 11 },  // A
        };

        static const char *estimate = "0123456789WX_%:";

        static const uint16_t ascii_map[] =
        {
            // 0/8  1/9     2/A     3/B     4/C     5/D     6/E     7/F
            0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, // 0x00 - 0x07
            0xffff, 0x0000, 0x0000, 0xffff, 0xffff, 0x0000, 0xffff, 0xffff, // 0x08 - 0x0f
            0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, // 0x10 - 0x17
            0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, // 0x18 - 0x1f
            0x0000, 0x0700, 0x0014, 0xffff, 0x00ed, 0xffff, 0xffff, 0x0010, // 0x20 - 0x27    !"#$%&'
            0x000e, 0x0070, 0x00b7, 0x0081, 0x0140, 0x0080, 0x0100, 0x0013, // 0x28 - 0x2f   ()*+,-./
            0x007e, 0x0030, 0x00da, 0x00f8, 0x00b4, 0x00ec, 0x00ee, 0x0038, // 0x30 - 0x37   01234567
            0x00fe, 0x00fc, 0x0600, 0x0600, 0x00b0, 0x00c0, 0x0086, 0x009a, // 0x38 - 0x3f   89:;<=>?
            0xffff, 0x00be, 0x00e6, 0x004e, 0x00f2, 0x00ce, 0x008e, 0x006e, // 0x40 - 0x47   @ABCDEFG
            0x00b6, 0x0006, 0x0072, 0x0027, 0x0066, 0x0038, 0x003e, 0x007e, // 0x48 - 0x4f   HIJKLMNO
            0x009e, 0x017e, 0x001e, 0x00ec, 0x00c6, 0x0076, 0x0062, 0x0070, // 0x50 - 0x57   PQRSTUVW
            0x0037, 0x00f4, 0x005b, 0x000e, 0x0025, 0x0070, 0x0004, 0x0040, // 0x58 - 0x5f   XYZ[\]^_
            0x0004, 0x00be, 0x00e6, 0x00c2, 0x00f2, 0x00de, 0x008e, 0x006e, // 0x60 - 0x67   `abcdefg
            0x00a6, 0x0006, 0x0070, 0x0027, 0x0046, 0x00a0, 0x00a2, 0x00e2, // 0x68 - 0x6f   hijklmno
            0x009e, 0x01e2, 0x0082, 0x00ec, 0x00c6, 0x0076, 0x0062, 0x0060, // 0x70 - 0x77   pqrstuvw
            0x0037, 0x00f4, 0x005b, 0x000e, 0x0036, 0x0070, 0x00e4, 0xffff, // 0x78 - 0x7f   xyz{|}~
            // Special cases: M, W, m, w
        };

        static const uint8_t ascii_bitmap[] =
        {
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x00,
            0x00, 0x00, 0x7E, 0x81, 0xA5, 0x81, 0x81, 0xBD, 0x99, 0x81, 0x81, 0x7E, 0x00, 0x00, 0x00,   //0x01,
            0x00, 0x00, 0x7E, 0xFF, 0xDB, 0xFF, 0xFF, 0xC3, 0xE7, 0xFF, 0xFF, 0x7E, 0x00, 0x00, 0x00,   //0x02,
            0x00, 0x00, 0x00, 0x00, 0x6C, 0xFE, 0xFE, 0xFE, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x00, 0x00,   //0x03,
            0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7C, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,   //0x04,
            0x00, 0x00, 0x00, 0x18, 0x3C, 0x3C, 0xE7, 0xE7, 0xE7, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00,   //0x05,
            0x00, 0x00, 0x00, 0x18, 0x3C, 0x7E, 0xFF, 0xFF, 0x7E, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00,   //0x06,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3C, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x07,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE7, 0xC3, 0xC3, 0xE7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,   //0x08,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x66, 0x42, 0x42, 0x66, 0x3C, 0x00, 0x00, 0x00, 0x00,   //0x09,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC3, 0x99, 0xBD, 0xBD, 0x99, 0xC3, 0xFF, 0xFF, 0xFF, 0xFF,   //0x0A,
            0x00, 0x00, 0x1E, 0x0E, 0x1A, 0x32, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x00, 0x00,   //0x0B,
            0x00, 0x00, 0x3C, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x00, 0x00,   //0x0C,
            0x00, 0x00, 0x3F, 0x33, 0x3F, 0x30, 0x30, 0x30, 0x30, 0x70, 0xF0, 0xE0, 0x00, 0x00, 0x00,   //0x0D,
            0x00, 0x00, 0x7F, 0x63, 0x7F, 0x63, 0x63, 0x63, 0x63, 0x67, 0xE7, 0xE6, 0xC0, 0x00, 0x00,   //0x0E,
            0x00, 0x00, 0x00, 0x18, 0x18, 0xDB, 0x3C, 0xE7, 0x3C, 0xDB, 0x18, 0x18, 0x00, 0x00, 0x00,   //0x0F,
            0x00, 0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFE, 0xF8, 0xF0, 0xE0, 0xC0, 0x80, 0x00, 0x00, 0x00,   //0x10,
            0x00, 0x02, 0x06, 0x0E, 0x1E, 0x3E, 0xFE, 0x3E, 0x1E, 0x0E, 0x06, 0x02, 0x00, 0x00, 0x00,   //0x11,
            0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00,   //0x12,
            0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00,   //0x13,
            0x00, 0x00, 0x7F, 0xDB, 0xDB, 0xDB, 0x7B, 0x1B, 0x1B, 0x1B, 0x1B, 0x1B, 0x00, 0x00, 0x00,   //0x14,
            0x00, 0x7C, 0xC6, 0x60, 0x38, 0x6C, 0xC6, 0xC6, 0x6C, 0x38, 0x0C, 0xC6, 0x7C, 0x00, 0x00,   //0x15,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFE, 0xFE, 0xFE, 0x00, 0x00, 0x00,   //0x16,
            0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x7E, 0x00, 0x00, 0x00,   //0x17,
            0x00, 0x00, 0x18, 0x3C, 0x7E, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,   //0x18,
            0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7E, 0x3C, 0x18, 0x00, 0x00, 0x00,   //0x19,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0C, 0xFE, 0x0C, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x1A,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xFE, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x1B, esc
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x1C,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6C, 0xFE, 0x6C, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x1D,
            0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7C, 0x7C, 0xFE, 0xFE, 0x00, 0x00, 0x00, 0x00,   //0x1E,
            0x00, 0x00, 0x00, 0x00, 0xFE, 0xFE, 0x7C, 0x7C, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,   //0x1F,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x20, ' '
            0x00, 0x00, 0x18, 0x3C, 0x3C, 0x3C, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,   //0x21, '!'
            0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x22, '"'
            0x00, 0x00, 0x00, 0x6C, 0x6C, 0xFE, 0x6C, 0x6C, 0x6C, 0xFE, 0x6C, 0x6C, 0x00, 0x00, 0x00,   //0x23, '#'
            0x00, 0x18, 0x18, 0x7C, 0xC6, 0xC2, 0xC0, 0x7C, 0x06, 0x86, 0xC6, 0x7C, 0x18, 0x18, 0x00,   //0x24, '$'
            0x00, 0x00, 0x00, 0x00, 0xC2, 0xC6, 0x0C, 0x18, 0x30, 0x60, 0xC6, 0x86, 0x00, 0x00, 0x00,   //0x25, '%'
            0x00, 0x00, 0x38, 0x6C, 0x6C, 0x38, 0x76, 0xDC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00,   //0x26, '&'
            0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x27, '''
            0x00, 0x00, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x18, 0x0C, 0x00, 0x00, 0x00,   //0x28, '('
            0x00, 0x00, 0x30, 0x18, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x18, 0x30, 0x00, 0x00, 0x00,   //0x29, ')'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3C, 0xFF, 0x3C, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x2A, '*'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x2B, '+'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00,   //0x2C, '
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x2D, '-'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,   //0x2E, '.'
            0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x80, 0x00, 0x00, 0x00,   //0x2F, '/'
            0x00, 0x00, 0x38, 0x6C, 0xC6, 0xC6, 0xD6, 0xD6, 0xC6, 0xC6, 0x6C, 0x38, 0x00, 0x00, 0x00,   //0x30, '0'
            0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7E, 0x00, 0x00, 0x00,   //0x31, '1'
            0x00, 0x00, 0x7C, 0xC6, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xC6, 0xFE, 0x00, 0x00, 0x00,   //0x32, '2'
            0x00, 0x00, 0x7C, 0xC6, 0x06, 0x06, 0x3C, 0x06, 0x06, 0x06, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x33, '3'
            0x00, 0x00, 0x0C, 0x1C, 0x3C, 0x6C, 0xCC, 0xFE, 0x0C, 0x0C, 0x0C, 0x1E, 0x00, 0x00, 0x00,   //0x34, '4'
            0x00, 0x00, 0xFE, 0xC0, 0xC0, 0xC0, 0xFC, 0x06, 0x06, 0x06, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x35, '5'
            0x00, 0x00, 0x38, 0x60, 0xC0, 0xC0, 0xFC, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x36, '6'
            0x00, 0x00, 0xFE, 0xC6, 0x06, 0x06, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00,   //0x37, '7'
            0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x38, '8'
            0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0x7E, 0x06, 0x06, 0x06, 0x0C, 0x78, 0x00, 0x00, 0x00,   //0x39, '9'
            0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,   //0x3A, ':'
            0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,   //0x3B, ';'
            0x00, 0x00, 0x00, 0x06, 0x0C, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x00, 0x00, 0x00,   //0x3C, '<'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x3D, '='
            0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x0C, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00,   //0x3E, '>'
            0x00, 0x00, 0x7C, 0xC6, 0xC6, 0x0C, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00,   //0x3F, '?'
            0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xDE, 0xDE, 0xDE, 0xDC, 0xC0, 0x7C, 0x00, 0x00, 0x00,   //0x40, '@'
            0x00, 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,   //0x41, 'A'
            0x00, 0x00, 0xFC, 0x66, 0x66, 0x66, 0x7C, 0x66, 0x66, 0x66, 0x66, 0xFC, 0x00, 0x00, 0x00,   //0x42, 'B'
            0x00, 0x00, 0x3C, 0x66, 0xC2, 0xC0, 0xC0, 0xC0, 0xC0, 0xC2, 0x66, 0x3C, 0x00, 0x00, 0x00,   //0x43, 'C'
            0x00, 0x00, 0xF8, 0x6C, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x6C, 0xF8, 0x00, 0x00, 0x00,   //0x44, 'D'
            0x00, 0x00, 0xFE, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x62, 0x66, 0xFE, 0x00, 0x00, 0x00,   //0x45, 'E'
            0x00, 0x00, 0xFE, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00,   //0x46, 'F'
            0x00, 0x00, 0x3C, 0x66, 0xC2, 0xC0, 0xC0, 0xDE, 0xC6, 0xC6, 0x66, 0x3A, 0x00, 0x00, 0x00,   //0x47, 'G'
            0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,   //0x48, 'H'
            0x00, 0x00, 0x3C, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00,   //0x49, 'I'
            0x00, 0x00, 0x1E, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x00, 0x00,   //0x4A, 'J'
            0x00, 0x00, 0xE6, 0x66, 0x66, 0x6C, 0x78, 0x78, 0x6C, 0x66, 0x66, 0xE6, 0x00, 0x00, 0x00,   //0x4B, 'K'
            0x00, 0x00, 0xF0, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x62, 0x66, 0xFE, 0x00, 0x00, 0x00,   //0x4C, 'L'
            0x00, 0x00, 0xC6, 0xEE, 0xFE, 0xFE, 0xD6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,   //0x4D, 'M'
            0x00, 0x00, 0xC6, 0xE6, 0xF6, 0xFE, 0xDE, 0xCE, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x00, 0x00,   //0x4E, 'N'
            0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x4F, 'O'
            0x00, 0x00, 0xFC, 0x66, 0x66, 0x66, 0x7C, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00,   //0x50, 'P'
            0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xD6, 0xDE, 0x7C, 0x0C, 0x0E, 0x00,   //0x51, 'Q'
            0x00, 0x00, 0xFC, 0x66, 0x66, 0x66, 0x7C, 0x6C, 0x66, 0x66, 0x66, 0xE6, 0x00, 0x00, 0x00,   //0x52, 'R'
            0x00, 0x00, 0x7C, 0xC6, 0xC6, 0x60, 0x38, 0x0C, 0x06, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x53, 'S'
            0x00, 0x00, 0x7E, 0x7E, 0x5A, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00,   //0x54, 'T'
            0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x55, 'U'
            0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x6C, 0x38, 0x10, 0x00, 0x00, 0x00,   //0x56, 'V'
            0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xD6, 0xD6, 0xD6, 0xFE, 0xEE, 0x6C, 0x00, 0x00, 0x00,   //0x57, 'W'
            0x00, 0x00, 0xC6, 0xC6, 0x6C, 0x7C, 0x38, 0x38, 0x7C, 0x6C, 0xC6, 0xC6, 0x00, 0x00, 0x00,   //0x58, 'X'
            0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00,   //0x59, 'Y'
            0x00, 0x00, 0xFE, 0xC6, 0x86, 0x0C, 0x18, 0x30, 0x60, 0xC2, 0xC6, 0xFE, 0x00, 0x00, 0x00,   //0x5A, 'Z'
            0x00, 0x00, 0x3C, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x3C, 0x00, 0x00, 0x00,   //0x5B, '['
            0x00, 0x00, 0x00, 0x80, 0xC0, 0xE0, 0x70, 0x38, 0x1C, 0x0E, 0x06, 0x02, 0x00, 0x00, 0x00,   //0x5C, '\'
            0x00, 0x00, 0x3C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x3C, 0x00, 0x00, 0x00,   //0x5D, ']'
            0x10, 0x38, 0x6C, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x5E, '^'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00,   //0x5F, '_'
            0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x60, '`'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00,   //0x61, 'a'
            0x00, 0x00, 0xE0, 0x60, 0x60, 0x78, 0x6C, 0x66, 0x66, 0x66, 0x66, 0x7C, 0x00, 0x00, 0x00,   //0x62, 'b'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC0, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x63, 'c'
            0x00, 0x00, 0x1C, 0x0C, 0x0C, 0x3C, 0x6C, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00,   //0x64, 'd'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xFE, 0xC0, 0xC0, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x65, 'e'
            0x00, 0x00, 0x38, 0x6C, 0x64, 0x60, 0xF0, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00,   //0x66, 'f'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0xCC, 0x78,   //0x67, 'g'
            0x00, 0x00, 0xE0, 0x60, 0x60, 0x6C, 0x76, 0x66, 0x66, 0x66, 0x66, 0xE6, 0x00, 0x00, 0x00,   //0x68, 'h'
            0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00,   //0x69, 'i'
            0x00, 0x00, 0x06, 0x06, 0x00, 0x0E, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3C,   //0x6A, 'j'
            0x00, 0x00, 0xE0, 0x60, 0x60, 0x66, 0x6C, 0x78, 0x78, 0x6C, 0x66, 0xE6, 0x00, 0x00, 0x00,   //0x6B, 'k'
            0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00,   //0x6C, 'l'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xEC, 0xFE, 0xD6, 0xD6, 0xD6, 0xD6, 0xC6, 0x00, 0x00, 0x00,   //0x6D, 'm'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xDC, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00,   //0x6E, 'n'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x6F, 'o'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xDC, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7C, 0x60, 0x60, 0xF0,   //0x70, 'p'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0x0C, 0x1E,   //0x71, 'q'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xDC, 0x76, 0x66, 0x60, 0x60, 0x60, 0xF0, 0x00, 0x00, 0x00,   //0x72, 'r'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0xC6, 0x60, 0x38, 0x0C, 0xC6, 0x7C, 0x00, 0x00, 0x00,   //0x73, 's'
            0x00, 0x00, 0x10, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x30, 0x30, 0x36, 0x1C, 0x00, 0x00, 0x00,   //0x74, 't'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00, 0x00, 0x00,   //0x75, 'u'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x18, 0x00, 0x00, 0x00,   //0x76, 'v'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xD6, 0xD6, 0xD6, 0xFE, 0x6C, 0x00, 0x00, 0x00,   //0x77, 'w'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0x6C, 0x38, 0x38, 0x38, 0x6C, 0xC6, 0x00, 0x00, 0x00,   //0x78, 'x'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x7E, 0x06, 0x0C, 0xF8,   //0x79, 'y'
            0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xCC, 0x18, 0x30, 0x60, 0xC6, 0xFE, 0x00, 0x00, 0x00,   //0x7A, 'z'
            0x00, 0x00, 0x0E, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0E, 0x00, 0x00, 0x00,   //0x7B, '{'
            0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00,   //0x7C, '|'
            0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0E, 0x18, 0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00,   //0x7D, '}'
            0x00, 0x00, 0x76, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x7E, '~'
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   //0x7F, delete
        };

        const w_class_t Indicator::metadata =           { "Indicator", &Widget::metadata };

        Indicator::Indicator(Display *dpy):
            Widget(dpy),
            sRows(&sProperties),
            sColumns(&sProperties),
            sShift(&sProperties),
            sTextGap(&sProperties),
            sLoop(&sProperties),
            sDarkText(&sProperties),
            sText(&sProperties),
            sType(&sProperties),
            sActive(&sProperties),
            sFont(&sProperties),
            sSpacing(&sProperties),
            sIPadding(&sProperties)
        {
            pClass      = &metadata;

            for (size_t i=0; i<IND_TOTAL; ++i)
                vColors[i].listener(&sProperties);

            nDWidth     = -1;
            nDHeight    = -1;
        }
        
        Indicator::~Indicator()
        {
            nFlags     |= FINALIZED;
        }

        status_t Indicator::init()
        {
            status_t result = Widget::init();
            if (result != STATUS_OK)
                return result;

            style::IndicatorColors *c = &vColors[style::INDICATOR_NORMAL];
            c->sColor.bind("color", &sStyle);
            c->sTextColor.bind("text.color", &sStyle);

            c = &vColors[style::INDICATOR_INACTIVE];
            c->sColor.bind("inactive.color", &sStyle);
            c->sTextColor.bind("inactive.text.color", &sStyle);

            sRows.bind("rows", &sStyle);
            sColumns.bind("columns", &sStyle);
            sShift.bind("text.shift", &sStyle);
            sTextGap.bind("text.gap", &sStyle);
            sLoop.bind("text.loop", &sStyle);
            sDarkText.bind("text.dark", &sStyle);
            sText.bind(&sStyle, pDisplay->dictionary());
            sType.bind("type", &sStyle);
            sActive.bind("active", &sStyle);
            sFont.bind("font", &sStyle);
            sSpacing.bind("spacing", &sStyle);
            sIPadding.bind("ipadding", &sStyle);

            return STATUS_OK;
        }

        style::IndicatorColors *Indicator::select_colors()
        {
            size_t flags = (sActive.get()) ? style::INDICATOR_NORMAL : style::INDICATOR_INACTIVE;
            return &vColors[flags];
        }

        void Indicator::property_changed(Property *prop)
        {
            Widget::property_changed(prop);

            // Self properties
            style::IndicatorColors *cols = select_colors();
            if (cols->property_changed(prop))
                query_draw();

            if (sActive.is(prop))
                query_draw();

            if (prop->one_of(sRows, sColumns, sType, sFont, sSpacing, sIPadding))
                query_resize();
            if (prop->one_of(sShift, sTextGap, sLoop, sDarkText, sText))
                query_draw();
        }

        void Indicator::calc_digit_size(ssize_t *w, ssize_t *h)
        {
            float fscaling  = lsp_max(0.0f, sScaling.get() * sFontScaling.get());

            switch (sType.get())
            {
                case INDICATOR_PIXEL:
                {
                    const float psize   = 1.6f * fscaling;
                    *w      = ceilf(8.0f * psize);
                    *h      = ceilf(15.0f * psize);
                    break;
                }

                case INDICATOR_MODERN:
                {
                    LSPString text;
                    ws::font_parameters_t fp;
                    ws::text_parameters_t tp;
                    sFont.get_parameters(pDisplay, fscaling, &fp);
                    *w  = 0;
                    *h  = fp.Height;

                    for (const char *c = estimate; *c != '\0'; ++c)
                    {
                        text.fmt_ascii("%c", *c);
                        sFont.get_text_parameters(pDisplay, &tp, fscaling, &text);
                        *w  = lsp_max(*w, ceilf(tp.Width));
                        *h  = lsp_max(*h, ceilf(tp.Height));
                    }
                    break;
                }

                case INDICATOR_SEGMENT:
                default:
                    *w      = ceilf(16.0f * fscaling);
                    *h      = ceilf(20.0f * fscaling);
                    break;
            }
        }

        void Indicator::draw_digit(ws::ISurface *s, float x, float y, size_t state, const lsp::Color &on, const lsp::Color &off)
        {
            const float fscaling    = lsp_max(0.0f, sScaling.get() * sFontScaling.get());
            const segment_t *seg    = segments;
            bool dark               = sDarkText.get();

            for (size_t i=0, m=1; i<11; ++i, m <<= 1, ++seg)
            {
                // Obtain color
                const lsp::Color *c =
                    (state & m) ? &on :
                    (dark) ? &off : NULL;
                if (c == NULL)
                    continue;

                const float dx      = x + seg->dx * fscaling + 0.5f;
                const float dy      = y + seg->dy * fscaling + 0.5f;

                const shape_t *sh   = &shapes[seg->shape];

                const point_t *p0 = &sh->p[0];
                for (size_t i=1; i <= sh->n; ++i)
                {
                    const point_t *p1 = &sh->p[i];
                    const point_t *p2 = &sh->p[i + 1];

                    s->fill_triangle(*c,
                        dx + float(p0->x) * fscaling, dy + float(p0->y) * fscaling,
                        dx + float(p1->x) * fscaling, dy + float(p1->y) * fscaling,
                        dx + float(p2->x) * fscaling, dy + float(p2->y) * fscaling
                    );
                }
            }
        }

        void Indicator::draw_pixel(ws::ISurface *s, float x, float y, char ch, const lsp::Color &on, const lsp::Color &off)
        {
            const float fscaling= lsp_max(0.0f, sScaling.get() * sFontScaling.get());
            const float psize   = 1.6f * fscaling;
            const float ppad    = 0.05f * psize;
            const float prsize  = psize - ppad * 2.0f;
            bool dark           = sDarkText.get();

            const uint8_t *p    = &ascii_bitmap[(ch & 0x7f) * 15];

            for (size_t dy=0; dy<15; ++dy)
            {
                uint8_t row         = *p++;
                const float py      = y + dy*psize + ppad;

                for (size_t dx=0; dx < 8; ++dx)
                {
                    const float px      = x + dx*psize + ppad;
                    if (row & (0x80 >> dx))
                        s->fill_rect(on, SURFMASK_NONE, 0.0f, px, py, prsize, prsize);
                    else if (dark)
                        s->fill_rect(off, SURFMASK_NONE, 0.0f, px, py, prsize, prsize);
                }
            }
        }

        void Indicator::draw_simple(ws::ISurface *s, float x, float y, char ch, const lsp::Color &on, const ws::font_parameters_t *fp)
        {
            float fscaling  = lsp_max(0.0f, sScaling.get() * sFontScaling.get());
            LSPString text;
            ws::text_parameters_t tp;

            text.fmt_ascii("%c", ch);
            sFont.get_text_parameters(s, &tp, fscaling, &text);

            sFont.draw(
                s, on,
                truncf(x + (nDWidth - tp.Width) * 0.5f),
                truncf(y + (nDHeight - fp->Height) + fp->Ascent),
                fscaling,
                &text);
        }

        void Indicator::size_request(ws::size_limit_t *r)
        {
            ssize_t dw, dh;
            float scaling   = lsp_max(0.0f, sScaling.get());
            size_t rows     = lsp_max(1, sRows.get());
            size_t cols     = lsp_max(1, sColumns.get());
            ssize_t spacing = (sSpacing.get() > 0) ? lsp_max(1.0f, sSpacing.get() * scaling) : 0;

            calc_digit_size(&dw, &dh);

            r->nMinWidth    = ceilf(dw * cols + spacing * (cols - 1));
            r->nMinHeight   = ceilf(dh * rows + spacing * (rows - 1));
            r->nMaxWidth    = r->nMinWidth;
            r->nMaxHeight   = r->nMinHeight;
            r->nPreWidth    = -1;
            r->nPreHeight   = -1;

            // Add internal padding
            sIPadding.add(r, scaling);
        }

        void Indicator::realize(const ws::rectangle_t *r)
        {
            calc_digit_size(&nDWidth, &nDHeight);
            Widget::realize(r);
        }

        uint8_t Indicator::get_char(const LSPString *str, size_t index)
        {
            // Compute the real index of character
            size_t length   = lsp_max(0, sTextGap.get()) + str->length();
            length          = lsp_max(1u, length);
            ssize_t shift   = sShift.get() + index;
            if (sLoop.get())
            {
                shift          %= ssize_t(length);
                if (shift < 0)
                    shift          += length;
            }

            // Get character
            lsp_wchar_t ch  = ((shift >= 0) && (size_t(shift) < str->length())) ? str->char_at(shift) : ' ';

            // Check ascii table match
            if (ch >= 0x80)
                ch          = 0;

            return uint8_t(ch);
        }

        void Indicator::draw(ws::ISurface *s)
        {
            float scaling   = lsp_max(0.0f, sScaling.get());
            float bright    = sBrightness.get();
            size_t rows     = lsp_max(1, sRows.get());
            size_t cols     = lsp_max(1, sColumns.get());
            size_t last     = rows * cols;
            ssize_t spacing = (sSpacing.get() > 0) ? lsp_max(1.0f, sSpacing.get() * scaling) : 0;
            bool dark       = sDarkText.get();
            const style::IndicatorColors *colors = select_colors();

            ws::rectangle_t xr;

            xr.nLeft        = 0;
            xr.nTop         = 0;
            xr.nWidth       = sSize.nWidth;
            xr.nHeight      = sSize.nHeight;

            // Prepare palette
            lsp::Color color(colors->sColor);
            lsp::Color on(colors->sTextColor);
            lsp::Color off(colors->sTextColor);

            off.blend(color, 0.05f);
            on.scale_lch_luminance(bright);
            off.scale_lch_luminance(bright);
            color.scale_lch_luminance(bright);

            // Draw glass
            s->clear(color);

            const bool aa = s->set_antialiasing(true);
            lsp_finally { s->set_antialiasing(aa); };
            sIPadding.enter(&xr, scaling);

            LSPString text;
            sText.format(&text);

            switch (sType.get())
            {
                case INDICATOR_MODERN:
                {
                    ws::font_parameters_t fp;
                    sFont.get_parameters(s, scaling, &fp);

                    for (size_t offset = 0, ich = 0; offset < last; )
                    {
                        // Get character
                        uint8_t ch  = get_char(&text, ich++);

                        // Output character
                        size_t col      = offset % cols;
                        size_t row      = offset / cols;

                        if (ch == '\n') // Need to fill up to end-of-line
                        {
                            if (dark)
                            {
                                for ( ; col < cols; ++col, ++offset)
                                    draw_simple
                                    (
                                        s,
                                        xr.nLeft + col*(nDWidth + spacing),
                                        xr.nTop  + row*(nDHeight + spacing),
                                        '8', off, &fp
                                    );
                            }
                        }
                        else
                        {
                            if (ch == ' ')
                            {
                                if (dark)
                                {
                                    draw_simple
                                    (
                                        s,
                                        xr.nLeft + col*(nDWidth + spacing),
                                        xr.nTop  + row*(nDHeight + spacing),
                                        '8', off, &fp
                                    );
                                }
                            }
                            else
                                draw_simple
                                (
                                    s,
                                    xr.nLeft + col*(nDWidth + spacing),
                                    xr.nTop  + row*(nDHeight + spacing),
                                    ch, on, &fp
                                );
                            ++offset;
                        }
                    }
                    break;
                } // INDICATOR_MODERN

                case INDICATOR_PIXEL:
                {
                    ws::font_parameters_t fp;
                    sFont.get_parameters(s, scaling, &fp);

                    for (size_t offset = 0, ich = 0; offset < last; )
                    {
                        // Get character
                        uint8_t ch  = get_char(&text, ich++);

                        // Output character
                        size_t col      = offset % cols;
                        size_t row      = offset / cols;

                        if (ch == '\n') // Need to fill up to end-of-line
                        {
                            if (dark)
                            {
                                for ( ; col < cols; ++col, ++offset)
                                    draw_pixel
                                    (
                                        s,
                                        xr.nLeft + col*(nDWidth + spacing),
                                        xr.nTop  + row*(nDHeight + spacing),
                                        ' ', on, off
                                    );
                            }
                        }
                        else
                        {
                            if (ch == ' ')
                            {
                                if (dark)
                                {
                                    draw_pixel
                                    (
                                        s,
                                        xr.nLeft + col*(nDWidth + spacing),
                                        xr.nTop  + row*(nDHeight + spacing),
                                        ch, on, off
                                    );
                                }
                            }
                            else
                                draw_pixel
                                (
                                    s,
                                    xr.nLeft + col*(nDWidth + spacing),
                                    xr.nTop  + row*(nDHeight + spacing),
                                    ch, on, off
                                );
                            ++offset;
                        }
                    }

                    break;
                } // INDICATOR_PIXEL

                case INDICATOR_SEGMENT:
                default:
                {
                    uint8_t unget = 0;
                    for (size_t offset = 0, ich = 0; offset < last; )
                    {
                        // Get character
                        uint8_t ch  = (unget > 0) ? unget : get_char(&text, ich++);

                        // Check for special case
                        if (!unget)
                        {
                            switch (ch)
                            {
                                case 'M':
                                    unget   = ch;
                                    ch      = 'N';
                                    break;
                                case 'm':
                                    unget   = ch;
                                    ch      = 'n';
                                    break;
                                case 'W':
                                    unget   = ch;
                                    ch      = 'U';
                                    break;
                                case 'w':
                                    unget   = ch;
                                    ch      = 'v';
                                    break;
                            }
                        }
                        else
                        {
                            unget       = 0;
                            if (ch == '\r')
                                continue;
                        }

                        // Save state
                        size_t state    = ascii_map[ch & 0xff];

                        // Lookup for extra characters
                        uint8_t ch2     = get_char(&text, ich);
                        switch (ch2)
                        {
                            case '.':
                            case ':':
                                state      |= ascii_map[ch2];
                                ++ich;
                                break;
                            default:
                                break;
                        }

                        // Output character
                        size_t col      = offset % cols;
                        size_t row      = offset / cols;

                        if (ch == '\n') // Need to fill up to end-of-line
                        {
                            for ( ; col < cols; ++col, ++offset)
                                draw_digit
                                (
                                    s,
                                    xr.nLeft + col*(nDWidth + spacing),
                                    xr.nTop  + row*(nDHeight + spacing),
                                    state, on, off
                                );
                        }
                        else
                        {
                            draw_digit
                            (
                                s,
                                xr.nLeft + col*(nDWidth + spacing),
                                xr.nTop  + row*(nDHeight + spacing),
                                state, on, off
                            );
                            ++offset;
                        }
                    }
                    break;
                } // INDICATOR_SEGMENT
            } // switch
        }

    } /* namespace tk */
} /* namespace lsp */
