/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/beta/cdf' );
import Beta = require( './../../../../../base/dists/beta/ctor' );
import entropy = require( './../../../../../base/dists/beta/entropy' );
import kurtosis = require( './../../../../../base/dists/beta/kurtosis' );
import logcdf = require( './../../../../../base/dists/beta/logcdf' );
import logpdf = require( './../../../../../base/dists/beta/logpdf' );
import mean = require( './../../../../../base/dists/beta/mean' );
import median = require( './../../../../../base/dists/beta/median' );
import mgf = require( './../../../../../base/dists/beta/mgf' );
import mode = require( './../../../../../base/dists/beta/mode' );
import pdf = require( './../../../../../base/dists/beta/pdf' );
import quantile = require( './../../../../../base/dists/beta/quantile' );
import skewness = require( './../../../../../base/dists/beta/skewness' );
import stdev = require( './../../../../../base/dists/beta/stdev' );
import variance = require( './../../../../../base/dists/beta/variance' );

/**
* Interface describing the `beta` namespace.
*/
interface Namespace {
	/**
	* Beta distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 0.5, 1.0, 1.0 );
	* // returns 0.5
	*
	* y = ns.cdf( 0.5, 2.0, 4.0 );
	* // returns ~0.813
	*
	* var myCDF = ns.cdf.factory( 0.5, 0.5 );
	*
	* y = myCDF( 0.8 );
	* // returns ~0.705
	*
	* y = myCDF( 0.3 );
	* // returns ~0.369
	*/
	cdf: typeof cdf;

	/**
	* Beta distribution.
	*/
	Beta: typeof Beta;

	/**
	* Returns the differential entropy of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns differential entropy
	*
	* @example
	* var v = ns.entropy( 1.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~-0.869
	*
	* @example
	* var v = ns.entropy( 8.0, 2.0 );
	* // returns ~-0.795
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 1.0, 1.0 );
	* // returns -1.2
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns ~0.082
	*
	* @example
	* var v = ns.kurtosis( 8.0, 2.0 );
	* // returns ~0.49
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Beta distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 5.0, 0.0, 4.0 );
	* // returns 0.0
	*
	* var mylogcdf = ns.logcdf.factory( 0.0, 10.0 );
	* y = mylogcdf( 0.5 );
	* // returns ~-1.938
	*
	* y = mylogcdf( 8.0 );
	* // returns ~-0.35
	*/
	logcdf: typeof logcdf;

	/**
	* Beta distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 0.5, 1.0, 1.0 );
	* // returns 0.0
	*
	* y = ns.logpdf( 0.5, 2.0, 4.0 );
	* // returns ~0.223
	*
	* var mylogpdf = ns.logpdf.factory( 0.5, 0.5 );
	*
	* y = mylogpdf( 0.8 );
	* // returns ~-0.228
	*
	* y = mylogpdf( 0.3 );
	* // returns ~-0.364
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 1.0, 1.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns 0.25
	*
	* @example
	* var v = ns.mean( 8.0, 2.0 );
	* // returns 0.8
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns median
	*
	* @example
	* var v = ns.median( 1.0, 1.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.median( 4.0, 12.0 );
	* // returns ~0.239
	*
	* @example
	* var v = ns.median( 8.0, 2.0 );
	* // returns ~0.820
	*
	* @example
	* var v = ns.median( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 2.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Beta distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.5, 1.0, 1.0 );
	* // returns ~1.297
	*
	* y = ns.mgf( 0.5, 2.0, 4.0 );
	* // returns ~1.186
	*
	* y = ns.mgf( 3.0, 2.0, 2.0 );
	* // returns ~5.575
	*
	* y = ns.mgf( -0.8, 4.0, 4.0 );
	* // returns ~0.676
	*
	* var myMGF = ns.mgf.factory( 0.5, 0.5 );
	*
	* y = myMGF( 0.8 );
	* // returns ~1.522
	*
	* y = myMGF( 0.3 );
	* // returns ~1.168
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns ~0.214
	*
	* @example
	* var v = ns.mode( 8.0, 2.0 );
	* // returns ~0.875
	*
	* @example
	* var v = ns.mode( 1.0, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, 0.8 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Beta distribution probability density function (PDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 0.5, 1.0, 1.0 );
	* // returns 1.0
	*
	* y = ns.pdf( 0.5, 2.0, 4.0 );
	* // returns 1.25
	*
	* var myPDF = ns.pdf.factory( 0.5, 0.5 );
	*
	* y = myPDF( 0.8 );
	* // returns ~0.796
	*
	* y = myPDF( 0.3 );
	* // returns ~0.695
	*/
	pdf: typeof pdf;

	/**
	* Beta distribution quantile function.
	*
	* @param p - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 2.0, 1.0 );
	* // returns ~0.894
	*
	* y = ns.quantile( 0.5, 4.0, 2.0 );
	* // returns ~0.686
	*
	* var myQuantile = ns.quantile.factory( 2.0, 2.0 );
	*
	* y = myQuantile( 0.8 );
	* // returns ~0.713
	*
	* y = myQuantile( 0.4 );
	* // returns ~0.5
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 1.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns ~0.529
	*
	* @example
	* var v = ns.skewness( 8.0, 2.0 );
	* // returns ~-0.829
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 1.0, 1.0 );
	* // returns ~0.289
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~0.105
	*
	* @example
	* var v = ns.stdev( 8.0, 2.0 );
	* // returns ~0.121
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a beta distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 1.0, 1.0 );
	* // returns ~0.083
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns ~0.011
	*
	* @example
	* var v = ns.variance( 8.0, 2.0 );
	* // returns ~0.015
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Beta distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
