﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mailmanager/MailManagerRequest.h>
#include <aws/mailmanager/MailManager_EXPORTS.h>

#include <utility>

namespace Aws {
namespace MailManager {
namespace Model {

/**
 * <p>The request to list archive export jobs in your account.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mailmanager-2023-10-17/ListArchiveExportsRequest">AWS
 * API Reference</a></p>
 */
class ListArchiveExportsRequest : public MailManagerRequest {
 public:
  AWS_MAILMANAGER_API ListArchiveExportsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListArchiveExports"; }

  AWS_MAILMANAGER_API Aws::String SerializePayload() const override;

  AWS_MAILMANAGER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The identifier of the archive.</p>
   */
  inline const Aws::String& GetArchiveId() const { return m_archiveId; }
  inline bool ArchiveIdHasBeenSet() const { return m_archiveIdHasBeenSet; }
  template <typename ArchiveIdT = Aws::String>
  void SetArchiveId(ArchiveIdT&& value) {
    m_archiveIdHasBeenSet = true;
    m_archiveId = std::forward<ArchiveIdT>(value);
  }
  template <typename ArchiveIdT = Aws::String>
  ListArchiveExportsRequest& WithArchiveId(ArchiveIdT&& value) {
    SetArchiveId(std::forward<ArchiveIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If NextToken is returned, there are more results available. The value of
   * NextToken is a unique pagination token for each page. Make the call again using
   * the returned token to retrieve the next page. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListArchiveExportsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of archive export jobs that are returned per call. You can
   * use NextToken to obtain further pages of archives. </p>
   */
  inline int GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  inline void SetPageSize(int value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = value;
  }
  inline ListArchiveExportsRequest& WithPageSize(int value) {
    SetPageSize(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_archiveId;

  Aws::String m_nextToken;

  int m_pageSize{0};
  bool m_archiveIdHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_pageSizeHasBeenSet = false;
};

}  // namespace Model
}  // namespace MailManager
}  // namespace Aws
